package org.eaglei.repository.servlet;

import java.net.URL;
import java.net.URLEncoder;
import java.util.List;
import java.util.ArrayList;
import java.io.InputStreamReader;
import java.io.StringReader;
import java.io.Reader;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.nio.charset.IllegalCharsetNameException;
import java.nio.charset.UnsupportedCharsetException;
import java.nio.charset.Charset;

import javax.servlet.http.HttpServlet;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.FileItem;
import org.apache.commons.fileupload.disk.DiskFileItemFactory;

import org.openrdf.repository.RepositoryException;
import org.openrdf.model.URI;

import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.ErrorSendingException;
import org.eaglei.repository.status.BadRequestException;

/**
 * Superclass for repo servlet classes, with shared utility methods.
 *
 * Also wraps the service() method with code that manages resources,
 * catches and processes the runtime errors used to signal HTTP status.
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class RepositoryServlet extends HttpServlet {

    private static Logger log = LogManager.getLogger(RepositoryServlet.class);

    // key in request attributes for cached file param list
    private static final String R_FILE_PARAMETERS = "org.eaglei.repository.Request.File.Parameters";

    private final static String CS = "UTF-8";  //char set for URL encoding

    /**
     * Simplified parameter-processing API:
     *  - abstract out the source of params from either Apache FileUpload
     *    for RFC 1867 compliant multipart/form-data or servlet for
     *    GET or regular form args.
     *  - check types and required params
     *
     * NOTE: automatically drop any arg values that are empty strings
     * since HTML forms tend to return an unwanted empty string
     * when input is left empty.  The ONLY case when an empty string
     * counts as a value is the isParameterPresent() call.
     */

    /**
     * Get a plain string parameter, including empty or whitespace string.
     * XXX needs javadoc
     */
    protected String getExactParameter(HttpServletRequest request, String name, boolean required)
    {
        String result = null;
        List<FileItem> fileParameters = getFileParameters(request);
        if (fileParameters != null) {
            for (FileItem fi : fileParameters) {
                if (fi.getFieldName().equals(name)) {
                    result = fi.getString();
                    if (result != null)
                        break;
                }
            }
        } else {
            result = request.getParameter(name);
        }
        if (required && result == null) {
            throw new BadRequestException("Missing required argument for: "+name);
        }
        return result;
    }

    /**
     * Get a plain string parameter, AND turn empty string into null.
     * XXX needs javadoc
     */
    protected String getParameter(HttpServletRequest request, String name, boolean required)
    {
        String result = getExactParameter(request, name, required);
        if (result != null && result.trim().length() == 0)
            result = null;
        return result;
    }

    /**
     * Get multiple params, return null for no params
     * XXX needs javadoc
     */
    protected String[] getParameters(HttpServletRequest request, String name, boolean required)
    {
        List<FileItem> fileParameters = getFileParameters(request);
        if (fileParameters != null) {
            List<String> lresult = new ArrayList<String>();
            for (FileItem fi : fileParameters) {
                if (fi.getFieldName().equals(name)) {
                    String s = fi.getString();
                    if (s.length() > 0)
                        lresult.add(s);
                }
            }
            if (lresult.isEmpty()) {
                if (required) {
                    throw new BadRequestException("Missing required argument(s) for: "+name);
                }
                return null;
            }
            return lresult.toArray(new String[lresult.size()]);
        } else {
            String[] result = request.getParameterValues(name);
            if (result != null && result.length == 1 && result[0].length() == 0)
                result = null;
            if (required && result == null) {
                throw new BadRequestException("Missing required argument(s) for: "+name);
            }
            return result;
        }
    }

    /**
     * return true when parameter is present, even if no value
     * XXX needs javadoc
     */
    protected boolean isParameterPresent(HttpServletRequest request, String name)
    {
        List<FileItem> fileParameters = getFileParameters(request);
        if (fileParameters != null) {
            for (FileItem fi : fileParameters) {
                if (fi.getFieldName().equals(name)) {
                    return true;
                }
            }
        } else {
            return request.getParameter(name) != null;
        }
        return false;
    }

    /**
     * XXX needs javadoc
     */
    protected boolean getParameterAsBoolean(HttpServletRequest request, String name, boolean defaultValue, boolean required)
    {
        return Utils.parseBooleanParameter(getParameter(request, name, required),
                                           name, required, defaultValue);
    }

    /**
     * XXX needs javadoc
     */
    protected URI getParameterAsURI(HttpServletRequest request, String name, boolean required)
    {
        return Utils.parseURI(getParameter(request, name, required), name, required);
    }

    /**
     * returns contnet-type of stream param or none if not there
     * XXX needs javadoc
     */
    protected String getParameterContentType(HttpServletRequest request, String name)
    {
        String result = null;
        List<FileItem> fileParameters = getFileParameters(request);
        if (fileParameters != null) {
            for (FileItem fi : fileParameters) {
                if (fi.getFieldName().equals(name)) {
                    return fi.getContentType();
                }
            }
        }
        return null;
    }

    /**
     * returns Reader on stream param or none if not there
     * XXX needs javadoc
     */
    protected Reader getParameterAsReader(HttpServletRequest request, String name, boolean required)
    {
        Reader result = null;
        List<FileItem> fileParameters = getFileParameters(request);
        if (fileParameters != null) {
            for (FileItem fi : fileParameters) {
                if (fi.getFieldName().equals(name)) {
                    try {
                        String csn = Utils.contentTypeGetCharset(fi.getContentType(), "UTF-8");
                        log.debug("For param="+name+", reading serialized RDF from stream with charset="+csn);
                        result = new InputStreamReader(fi.getInputStream(), Charset.forName(csn));
                    } catch  (IOException e) {
                        throw new BadRequestException("Failed reading argument="+name+", I/O Exception: "+e);
                    } catch  (IllegalCharsetNameException e) {
                        throw new BadRequestException("Failed reading argument="+name+", Illegal character set name in content-type spec: "+e);
                    } catch  (UnsupportedCharsetException e) {
                        throw new BadRequestException("Failed reading argument="+name+", Unsupported character set name in content-type spec: "+e);
                    }
                    break;
                }
            }
        } else {
            String pv = request.getParameter(name);
            if (pv != null) {
                result = new StringReader(pv);
            }
        }
        if (required && result == null) {
            throw new BadRequestException("Missing required argument for: "+name);
        }
        return result;
    }

    /**
     *
     * XXX needs javadoc
     */
    protected Enum getParameterAsKeyword(HttpServletRequest request,
            String name, Class argType, Enum defaultValue, boolean required)
    {
        return Utils.parseKeywordArg(argType, getParameter(request, name, required), name, required, defaultValue);
    }

    /**
     * returns FileParam list or null if not that kind of req (or no params)
     * cache the result in the request since servlet object is persistent!
     * XXX needs javadoc
     */
    private List<FileItem> getFileParameters(HttpServletRequest request)
    {
        // the params if this req was RFC 1867 compliant multipart/form-data
        List<FileItem> fileParameters = (List<FileItem>)request.getAttribute(R_FILE_PARAMETERS);

        if (fileParameters == null) {
            try {
                request.setCharacterEncoding("UTF-8");
                if (ServletFileUpload.isMultipartContent(request)) {
                        ServletFileUpload upload = new ServletFileUpload();
                        upload.setFileItemFactory(new DiskFileItemFactory(100000,
                          (File)getServletConfig().getServletContext().getAttribute("javax.servlet.context.tempdir")));
                        fileParameters = upload.parseRequest(request);
                } else {
                    fileParameters = new ArrayList<FileItem>();
                }
            } catch (UnsupportedEncodingException e) {
                log.error(e);
                throw new BadRequestException("Unacceptable encoded character in argument data", e);
            } catch  (FileUploadException e) {
                log.error(e);
                throw new BadRequestException("failed parsing multipart request", e);
            }
            request.setAttribute(R_FILE_PARAMETERS, fileParameters);
        }
        return fileParameters.isEmpty() ? null : fileParameters;
    }



    /**
     * Wrap service() with error-handling, logging, and cleanup of any
     * Sesame repository connection left open.
     *
     * {@inheritDoc}
     */
    @Override
    protected void service(HttpServletRequest req, HttpServletResponse resp)
        throws ServletException, IOException
    {
        String reqURI = req.getRequestURI();
        boolean finished = false;
        long startMs = System.currentTimeMillis();
        try {
            // make log files easier to trace
            if (log.isDebugEnabled())
                log.debug("============== Starting Request "+reqURI);
            super.service(req, resp);
            finished = true;

        // Servlets can throw BadRequetException to elicit a
        // 400 response, since ServletException results in 500..
        // Same mechanism for ForbiddenEx. and NotFoundEx.
        } catch (ErrorSendingException e) {
            // pass this exception to error.jsp through attribute
            req.setAttribute("org.eaglei.repository.error.exception", e);
            if (resp.isCommitted()) {
                log.fatal("Exception thrown after response was committed so the status MAY FALSELY INDICATE SUCCESS.");
            }
            try {
                resp.sendError(e.getStatus(), e.getMessage());
            } catch (IllegalStateException ie) {
                log.error("Cannot convert Status Exception to appropriate status because response is in wrong state.  Original exception="+e.toString(), ie);
            }
        } catch (ServletException e) {
            if (resp.isCommitted()) {
                log.fatal("Exception thrown after response was committed so the status MAY FALSELY INDICATE SUCCESS.");
            }
            throw e;
        } catch (RuntimeException e) {
            if (resp.isCommitted()) {
                log.fatal("Exception thrown after response was committed so the status MAY FALSELY INDICATE SUCCESS.");
            }
            throw e;
        } finally {
            // rollback now, prevents Sesame from logging errors at rc close.
            if (!finished) {
                try {
                    WithRepositoryConnection.get(req).rollback();
                } catch (RepositoryException re) {
                    log.warn("Failed in rollback after error: "+re);
                }
            }
            if (log.isDebugEnabled())
                log.debug(String.format("============== Ending Request %s (%,d mSec elapsed)",reqURI, System.currentTimeMillis()-startMs));
        }
    }

    /**
     * Detect and manage the redirect to a JSP page whose form action invoked
     * this servlet.  Upon success, the servlet wants to present the JSP
     * again with its new state and a messaeg (the "message" arg) explaining
     * what was done, to indicate success.
     *
     * This ONLY redirects when the referrer was a JSP, i.e. the URL has
     * ".jsp" in it.  It expects the rererring JSP to be in the repo admin
     * UI so it follows those conventions (message, other args).
     *
     * WARNING, note that the 'more' vararg is expected to be an even
     * number of args, alternating arg name and value.  The values of all
     * args are escaped by this function.
     *
     * @param request
     * @param response
     * @param message value of 'message' arg to send jsp
     * @param passThru names of servlet params to pass through to jsp request
     * @param more additional params; alternating param name and value (unencoded) to send to JSP
     */
    protected void redirectToJSP(HttpServletRequest request, HttpServletResponse response,
                                 String message, String passThru[], String... more)
        throws ServletException, java.io.IOException
    {
        // if executing a request for a .jsp page, redirect back to it:
        // XXX this is pretty sketchy logic, but it works..
        String referrer = request.getHeader("Referer");
        if (referrer != null && referrer.indexOf(".jsp") >= 0) {
            log.debug("Got referrer = "+referrer);
            URL from = new URL(referrer);
            StringBuilder redir = new StringBuilder();
            redir.append(from.getProtocol()).append("://").append(from.getHost());
            if (from.getPort() >= 0)
                redir.append(":").append(String.valueOf(from.getPort()));
            redir.append(from.getPath());
            redir.append("?message=").append(URLEncoder.encode(message, CS));
            for (String p : passThru) {
                String pv = getExactParameter(request, p, false);
                if (pv != null)
                    redir.append("&").append(p).append("=").append(URLEncoder.encode(pv, CS));
            }
            for (int i = 0; i < more.length; i+= 2) {
                redir.append("&").append(more[i]).append("=").append(URLEncoder.encode(more[i+1], CS));
            }
            if (log.isDebugEnabled())
                log.debug("Redirecting to = "+redir.toString());
            response.sendRedirect(response.encodeRedirectURL(redir.toString()));
        }
    }
}
