package org.eaglei.repository.servlet.admin;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.model.URI;
import org.openrdf.repository.RepositoryConnection;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.Role;
import org.eaglei.repository.servlet.RepositoryServlet;
import org.eaglei.repository.servlet.WithRepositoryConnection;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.NotFoundException;

/**
 * Create and/or modify a Role.  This servlet exists primarily
 * to serve the Edit Role page in the admin JSP UI.  Of course
 * it can be used by itself, but beware of weird behavior when its
 * referrer is a .JSP URL - it will indicate success by redirecting back
 * to the JSP.
 *
 * Method: POST only.
 *
 *   REQUIRED ARGS:
 *    uri = URI of the object
 *    action = create | update | delete
 *   optional:
 *    label   (requred to create)
 *    comment
 *
 * @author Larry Stone
 * Started March 2011
 */
public class UpdateRole extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(UpdateRole.class);

    // allowable values for Action arg
    public enum Action { create, update, delete };

    // args to pass through to JSP
    private static final String PASS_THRU_ARGS[] = {};

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        URI uri = getParameterAsURI(request, "uri", false);
        Action action = (Action)getParameterAsKeyword(request, "action", Action.class, null, true);
        String label = getParameter(request, "label", false);
        String comment = getParameter(request, "comment", false);

        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("This service is only for use by administrators.");

        // sanity-check args
        if (label == null)
            label = "";
        if (comment == null)
            comment = "";
        if (uri == null && action != Action.create) {
            throw new BadRequestException("Value for 'uri' is required when action is not create.");
        }
        if (action == Action.create && label == null) {
            throw new BadRequestException("Value for 'label' is required when action is create.");
        }

        RepositoryConnection rc = WithRepositoryConnection.get(request);
        String message = "";
        Role r = null;

        if (action == Action.create) {
            r = Role.create(request, uri, label, comment, false);
            uri = r.getURI();
            message = "Created new Role.";

        } else {
            r = Role.find(request, uri);
            if (r == null)
                throw new NotFoundException("No such role, uri="+uri);

            if (action == Action.update) {
                r.setLabel(request, label);
                r.setComment(request, comment);
                message = "Modified role \""+r.getLabel()+"\"";
             
            } else if (action == Action.delete) {
                r.delete(request);
                message = "Deleted role \""+r.getLabel()+"\"";
             
            }
        }
        if (r != null)
            r.commit(request);

        redirectToJSP(request, response, message, PASS_THRU_ARGS, "uri", uri.stringValue());
    }
}
