package org.eaglei.repository.format;

import java.io.PrintStream;
import java.io.OutputStream;
import java.io.UnsupportedEncodingException;
import java.util.List;

import org.openrdf.model.Value;
import org.openrdf.query.BindingSet;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultWriter;
import org.openrdf.query.TupleQueryResultHandlerException;

/**
 * Tuple query results writer for plain text format.
 * Very simple and stupid "TSV" (tab-separated values)
 * format, not even checking for embedded tabs in values.
 * Perhaps it would be better to have a more human-readable
 * text format (e.g. columns aligned by whitespace, maybe borders)
 * but this was quick and expedient.
 *
 * @author lcs
 * @version $Id: $
 */
public class SPARQLTextWriter implements TupleQueryResultWriter
{
    /** Constant <code>FORMAT</code> describes this format */
    public static final TupleQueryResultFormat FORMAT =
        new TupleQueryResultFormat("Tab-Separated Values", "text/plain", ".txt");

    private PrintStream out = null;

    private OutputStream outStream = null;

    private List<String> columns = null;

    /**
     * <p>Default constructor for SPARQLTextWriter.  Assumes UTF-8 character set in output.</p>
     *
     * @param out desintation stream, a {@link java.io.OutputStream} object.
     */
    public SPARQLTextWriter(OutputStream out)
    {
        super();
        this.outStream = out;
    }

    /**
     * Get description of format
     * @return
     */
    public TupleQueryResultFormat getTupleQueryResultFormat()
    {
        return FORMAT;
    }

    /**
     * Start streaming result for given bindings
     * @param bindingNames
     * @throws TupleQueryResultHandlerException
     */
    public void startQueryResult(List<String> bindingNames)
                          throws TupleQueryResultHandlerException
    {
        try {
            out = new PrintStream(outStream, false, "UTF-8");
        } catch (UnsupportedEncodingException e) {
            throw new TupleQueryResultHandlerException(e);
        }
        columns = bindingNames;
        boolean first = true;
        for (String bn : bindingNames) {
            out.print(first ? bn : "\t"+bn);
            first = false;
        }
        out.println();
    }

    /**
     * Finish streaming result
     * @throws TupleQueryResultHandlerException
     */
    public void endQueryResult()
                        throws TupleQueryResultHandlerException
    {
    }

    /**
     * Add a solution to the stream
     * @param bs
     * @throws TupleQueryResultHandlerException
     */
    public void handleSolution(BindingSet bs)
                        throws TupleQueryResultHandlerException
    {
        boolean first = true;
        for (String bn : columns) {
            if (!first)
                out.print("\t");
            if (bs.hasBinding(bn)) {
                Value bsv = bs.getValue(bn);
                out.print(bsv == null ? "" : bsv.toString());
            }
            first = false;
        }
        out.println();
    }
}
