package org.eaglei.repository.model;

import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import java.util.ArrayList;
import java.util.List;
import java.util.HashMap;
import java.util.Map;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.impl.LiteralImpl;
import org.openrdf.model.Resource;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.XMLSchema;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.HasContentCache;

/**
 * <pre>
 * Named Graph object model, collects metadata about the named graph
 * stored in repo's intenal metadata graph.  NG is implemented as a Sesame
 * context, named by the "name" URI.  It has these attributes:
 *  * name - URI of the context
 *  * label - text string to describe graph to users
 *  * graph type - enum identifying hte purpose of its contents
 *  * provenance metadata
 *  * access control (admin metadata)
 *
 * </pre>
 * @author Larry Stone
 * Started June 7, 2010
 */
@HasContentCache
public final class NamedGraph extends WritableObjectModel implements Comparable<NamedGraph>
{
    private static Logger log = LogManager.getLogger(NamedGraph.class);

    /** cache of NGTs, by URI */
    private static volatile Map<URI,NamedGraph> uriToNG = null;

    /**
     * SPARQL query to get NG properties & admin metadata.  Bind
     * "?namedGraphURI" to query for ONE named graph, otherwise it gets ALL.
     * ..BUT this does not necessarily match Sesame's list of context, it
     * is still necessary to compre them.
     * XXX NOTE that the HAS_READ_ACCESS clause is a kludgy optimization
     *   that violates the access control abstraction barrier.
     */
    private static final String ngQuery =
          "SELECT DISTINCT * WHERE { ?namedGraphURI a <"+REPO.NAMED_GRAPH+"> . \n"+
          " OPTIONAL { ?namedGraphURI <"+RDFS.LABEL+"> ?namedGraphLabel } \n"+
          " OPTIONAL { ?namedGraphURI <"+REPO.NG_TYPE+"> ?typeURI } \n"+
          " OPTIONAL { ?namedGraphURI <"+REPO.HAS_READ_ACCESS+"> ?anon \n"+
          "            FILTER( ?anon = <"+REPO.ROLE_ANONYMOUS+"> )}}";

    // instance variables
    private URI name;               // URI identifying the context.
    private String label;           // human-readable text label
    private NamedGraphType type;              // type of graph
    private boolean anonAccess;     // is is publically readable?
    private boolean hasMetadata = false;  // is there even NG metadata?

    /** full constructor for NG instance with metadata */
    private NamedGraph(URI name, String label, URI type, boolean anonAccess)
    {
        super();
        this.name = name;
        this.label = label;
        this.type = NamedGraphType.parse(type);
        this.anonAccess = anonAccess;
        this.hasMetadata = true;
    }

    /** constructor for NG without metadata, e.g. from Sesame context list */
    private NamedGraph(URI name)
    {
        super();
        this.name = name;
        this.hasMetadata = false;
    }

    /**
     * Invalidate the entire NG cache, called when RDF has changed.
     * It's cheap enough to reload, and changes are very infrequent.
     */
    public static void decache()
    {
        synchronized (NamedGraph.class) {
            uriToNG = null;
        }
        log.debug("Cleared global NamedGraph cache map.");
    }

    /**
     * Invalidate local cache after changes to RDF.
     */
    public void decacheInstance()
    {
        decache();
    }

    /**
     * This gets called to force out changes, but not "commit" them.
     * If anything actually changed, decache.
     * @param request the HTTP request object from the servlet
     */
    public void update(HttpServletRequest request)
        throws ServletException
    {
        if (isDirty())
            decache();
    }

    // return the cached map, filling it first if necessary
    private static Map<URI,NamedGraph> getMap(HttpServletRequest request)
        throws ServletException
    {
        // this looks weird but is necessary to minimize sync block.
        synchronized (NamedGraph.class) {
            if (uriToNG != null)
                return uriToNG;
            uriToNG = new HashMap<URI,NamedGraph>();
        }
        try {
            log.debug("Filling the global NamedGraph cache..");
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, ngQuery);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(true);
            SPARQL.evaluateTupleQuery(ngQuery, q, new NamedGraphHandler(uriToNG));

            // Second pass, check for Sesame contexts that don't have NG metadata.
            // If one of these exists, it implies something was writing into
            // an "undeclared" named graph.
            RepositoryResult<Resource> rr = null;
            try {
                rr = rc.getContextIDs();
                while (rr.hasNext()) {
                    Resource ctx = rr.next();
                    if (ctx instanceof URI && !uriToNG.containsKey((URI)ctx)) {
                        if (log.isDebugEnabled())
                            log.debug("Found 'undocumented' named graph (context), name="+ctx);
                        uriToNG.put((URI)ctx, new NamedGraph((URI)ctx));
                    }
                }
            } finally {
                rr.close();
            }
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return uriToNG;
    }

    /**
     * Get an iterable collection of all known named graphs AND Contexts.
     * If there is no Named Graph metadata for a Sesame context, its
     * isManaged() method will return false.
     * Note that named graphs are ONLY listed ONCE even if there is duplicate
     * metadata.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all named graphs in a {@link java.util.Collection} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<NamedGraph> findAll(HttpServletRequest request)
        throws ServletException
    {
        return new ArrayList<NamedGraph>(getMap(request).values());
    }

    /**
     * Returns a named graph object, or null if this is not a valid
     * named graph OR context.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name name of named-graph or context, a {@link org.openrdf.model.URI} object.
     * @return a {@link org.eaglei.repository.NamedGraph} object representing named graph, or null if no graph or context exists.
     * @throws javax.servlet.ServletException if any problems
     */
    public static NamedGraph find(HttpServletRequest request, URI name)
        throws ServletException
    {
        return findOrCreate(request, name, false);
    }

    /**
     * Returns a named graph object, or null if this is not a valid
     * named graph OR context and createp was false.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param name name of named-graph or context, a {@link org.openrdf.model.URI} object.
     * @param createp a boolean, true if graph should be created
     * @return a {@link org.eaglei.repository.NamedGraph} object representing named graph, or null if no graph or context exists UNLESS createp is true.
     * @throws javax.servlet.ServletException if any problems.
     */
    public static NamedGraph findOrCreate(HttpServletRequest request, URI name, boolean createp)
        throws ServletException
    {
        Map<URI,NamedGraph> ngs = getMap(request);
        NamedGraph result = ngs.get(name);
        if (result == null) {
            if (createp) {
                result = new NamedGraph(name);
                      // this will also decache the map..
            } else {
                log.warn("Failed to find named graph or context for name="+name);
            }
        }
        return result;
    }

    /**
     * <p>Getter for the field <code>name</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getURI()
    {
        return name;
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getLabel()
    {
        if (label != null)
            return label;
        else
            return name.getLocalName();
    }

    /**
     * <p>Getter for the field <code>type</code>.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public NamedGraphType getType()
    {
        return type;
    }

    /**
     * <p>Getter for the field <code>type</code>, but returns URI form.</p>
     *
     * @return a {@link org.openrdf.model.URI} object.
     */
    public URI getTypeURI()
    {
        return type == null ? null : type.getURI();
    }

    /**
     * <p>Getter for the field <code>typeLabel</code>.</p>
     *
     * @return a {@link java.lang.String} object.
     */
    public String getTypeLabel()
    {
        return type == null ? null : type.getLabel();
    }

    /**
     * Can be read by anonymous user (i.e. general public) if true.
     *
     * @return a boolean, true if graph is publically readable.
     */
    public boolean isAnonymousReadable()
    {
        return anonAccess;
    }

    /**
     * Return the number of non-inferred statements in the named graph.
     *
     * @return count of non-inferred statements in the named graph.
     */
    public long getSize(HttpServletRequest request)
        throws ServletException
    {
        try {
            return WithRepositoryConnection.get(request).size(name);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    /**
     * When true, graph has metadata and is managed by repository.
     * False for unmanaged sesame contexts.
     *
     * @return a boolean, true when graph is known NamedGraph.
     */
    public boolean isManaged()
    {
        return hasMetadata;
    }

    /**
     * <p>Setter for the field <code>label</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nl a {@link java.lang.String} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setLabel(HttpServletRequest request, String nl)
        throws ServletException
    {
        setMetadataInternal(request, RDFS.LABEL, new LiteralImpl(nl, XMLSchema.STRING));
        label = nl;
    }

    /**
     * <p>Setter for the field <code>type</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nt a {@link org.eaglei.repository.NamedGraph.NamedGraphType} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setType(HttpServletRequest request, URI nt)
        throws ServletException
    {
        NamedGraphType ntt = NamedGraphType.parse(nt);
        if (ntt != null)
            setType(request, ntt);
    }

    /**
     * <p>Setter for the field <code>type</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param nt a {@link org.openrdf.model.URI} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void setType(HttpServletRequest request, NamedGraphType nt)
        throws ServletException
    {
        setMetadataInternal(request, REPO.NG_TYPE, nt.getURI());
        type = nt;
    }

    /** set metadata and also ensure there is a rdf:type statement */
    private void setMetadataInternal(HttpServletRequest request, URI property, Value newVal)
        throws ServletException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            boolean hasType = rc.hasStatement(name, RDF.TYPE, REPO.NAMED_GRAPH, false, REPO.NG_INTERNAL);
            if (hasType && rc.hasStatement(name, property, newVal, false, REPO.NG_INTERNAL)) {
                log.debug("Nothing to set, graph="+name+" already has "+property+" = "+newVal);
            } else {
                if (!hasType) {
                    rc.add(name, RDF.TYPE, REPO.NAMED_GRAPH, REPO.NG_INTERNAL);
                    hasMetadata = true;
                }
                rc.remove(name, property, null, REPO.NG_INTERNAL);
                rc.add(name, property, newVal, REPO.NG_INTERNAL);
                setDirty(true);
                log.debug("Setting graph="+name+",  "+property+" = "+newVal);
            }
        } catch (OpenRDFException e) {
            throw new ServletException(e);
        }
    }


    /**
     * Consider two NGs equal if the names are equal, never mind the metadata.
     * {@inheritDoc}
     */
    public boolean equals(Object b)
    {
        return b instanceof NamedGraph && ((NamedGraph)b).name.equals(name);
    }

    /** {@inheritDoc} */
    public int hashCode()
    {
        return name.hashCode();
    }

    /** {@inheritDoc} */
    public int compareTo(NamedGraph o)
    {
        return name.toString().compareTo(o.name.toString());
    }

    /**
     * query result handler to populate result list with NamedGraph objects
     */
    private static final class NamedGraphHandler extends TupleQueryResultHandlerBase
    {
        // key is graph URI for easy comparison and lookup of duplicates
        private Map<URI,NamedGraph> result = null;
        private boolean checkDuplicates = false;

        private NamedGraphHandler(Map<URI,NamedGraph> result)
        {
            super();
            NamedGraphHandler.this.result = result;

            // if we're starting out with some graphs in map, ignore
            // "duplicates" since there would be false positives.
            checkDuplicates =  result.isEmpty();
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value ngURI = bs.getValue("namedGraphURI");
            Value ngLabel = bs.getValue("namedGraphLabel");
            Value typeURI = bs.getValue("typeURI");
            Value anon = bs.getValue("anon");
            if (!(ngURI instanceof URI))
                throw new TupleQueryResultHandlerException(
                    "Should not get null or non-URI result in NamedGraphHandler: "+ngURI);

            // check for duplicate metadata on same graph URI, should NOT happen
            else if (checkDuplicates && result.containsKey(ngURI))
                log.warn("There are MULTIPLE named graph metadata entries for graph="+ngURI+", skipping extra: label="+ngLabel+", type="+typeURI);

            // looks like a good entry
            else {
                String ngl = (ngLabel instanceof Literal) ?
                           ((Literal)ngLabel).getLabel() : ngURI.toString();
                boolean anonAccess = (anon != null);
                result.put((URI)ngURI, new NamedGraph((URI)ngURI, ngl, (URI)typeURI, anonAccess));
            }
        }
    }
}
