package org.eaglei.repository.model;

import java.util.Map;
import java.util.HashMap;
import java.util.ArrayList;
import java.util.List;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletException;

import org.openrdf.model.URI;
import org.openrdf.model.Statement;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;

import org.eaglei.repository.HasContentCache;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.vocabulary.REPO;

/**
 * This class is both an enumeration and an ImmutableObjectModel
 * representing the allowable Types of Named Graphs.
 *
 * Although the types of named graphs are part of the repository's
 * internal ontology and thus appear that they might be "easily"
 * changed, in reality, the semantics of named graph types are
 * thoroughly baked into the repo logic such that if they are changed or
 * extended, updating this enum is the least of the work required.
 *
 * The symbolic name is also how the named graph type appears in the
 * API, at least, where a keyword is used instead of the URI.
 *
 * @author Larry Stone
 * Started June 7, 2010
 */
@HasContentCache
public enum NamedGraphType /* extends ImmutableObjectModel */
{
    /**
     * Types:
     */
    ontology(REPO.NGTYPE_ONTOLOGY),
    metadata(REPO.NGTYPE_METADATA),
    workspace(REPO.NGTYPE_WORKSPACE),
    published(REPO.NGTYPE_PUBLISHED),
    internal(REPO.NGTYPE_INTERNAL);

    private static Logger log = LogManager.getLogger(NamedGraphType.class);

    /** the associated URI in the ontology */
    private URI uri = null;

    /** the associated rdfs:label in the ontology */
    private String label = null;

    /** cache of NGTs, by URI */
    private static volatile Map<URI,NamedGraphType> uriToNGT = null;

    /**
     * Invalidate the NGT cache, called when RDF has changed.
     */
    public static void decache()
    {
        synchronized (NamedGraphType.class) {
            uriToNGT = null;
        }
    }

    private static Map<URI,NamedGraphType> getMap(HttpServletRequest request)
    {
        // this looks weird but is necessary to minimize sync block.
        synchronized (NamedGraphType.class) {
            if (uriToNGT != null)
                return uriToNGT;
            uriToNGT = new HashMap<URI,NamedGraphType>();
        }
        try {
            log.debug("Filling the NamedGraphType cache..");
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            for (NamedGraphType ngt : NamedGraphType.values()) {
                URI subject = ngt.getURI();
                uriToNGT.put(subject, ngt);
                RepositoryResult<Statement> rr = rc.getStatements(subject, RDFS.LABEL, null, false, REPO.NG_REPO_ONTOLOGY);
                try {
                    if (rr.hasNext()) {
                        ngt.label = Utils.valueAsString(rr.next().getObject());
                        log.debug("Found label for uri="+ngt.getURI()+", label=\""+ngt.label+"\"");
                    } else {
                        log.warn("Cannot find label, substituting default for uri="+ngt.getURI());
                        ngt.label = ngt.defaultLabel();
                    }
                } finally {
                    rr.close();
                }
            }
        } catch (RepositoryException e) {
            log.error(e);
            throw new InternalServerErrorException(e);
        }
        return uriToNGT;
    }

    /** canonical constructor */
    private NamedGraphType(URI u) {
        uri = u;
    }

    /** Accessor */
    public URI getURI() {
        return uri;
    }

    /** Interpret string as one of the enumerated type names */
    public static NamedGraphType parse(String v) {
        try {
            return valueOf(v);
        } catch (IllegalArgumentException e) {
            return null;
        }
    }

    /** Interpret URI as one of the enumerated type names */
    public static NamedGraphType parse(URI v) {
        for (NamedGraphType t : values()) {
            if (t.uri.equals(v)) {
                return t;
            }
        }
        return null;
    }

    // default stand-in for label from ontology
    private String defaultLabel()
    {
        String result = toString();
        return String.format("%C%s", result.charAt(0), result.substring(1));
    }

    /** Get pretty title to include in e.g. a menu */
    public String getLabel()
    {
        return label == null ? defaultLabel() : label;
    }

    /**
     * Get NGT from the fixed map, we do not expect to update the ontology.
     * {@inheritDoc}
     */
    public static NamedGraphType find(HttpServletRequest request, URI subject)
        throws ServletException
    {
        return getMap(request).get(subject);
    }

    /**
     * Get list of all types.
     * Almost never gets called, so inefficient implementation is ok
     * {@inheritDoc}
     */
    public static List<NamedGraphType> findAll(HttpServletRequest request)
        throws ServletException
    {
        return new ArrayList(getMap(request).values());
    }
}
