package org.eaglei.repository.model;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.HashMap;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;
import org.eaglei.repository.Configuration;
import org.eaglei.repository.HasContentCache;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Value;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryException;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.ValueFactory;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.BindingSet;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.rid.RIDGenerator;

/**
 * Role object model, reflects the :Role object in RDF database.
 *
 * A role represents an entity that can be granted privileges in the access
 * control system.  Users who are <i>members</i> of a Role get all its
 * grants.  Roles are "flat" objects, i.e. there is NO hierarchy of Roles --
 * although it would be convenient, it is too difficult to implement given
 * the current inferencing support available in the RDF database.
 *
 * Named Graph usage: Role looks for statements in ALL graphs, although
 *  they should really only be found in the :NG_Internal graph.
 *
 * @author Larry Stone
 * Started April 26, 2010
 */
@HasContentCache
public final class Role extends WritableObjectModel implements Comparable
{
    private static Logger log = LogManager.getLogger(Role.class);

    private URI uri;                  // subject: required
    private String label = null;      // value of rdfs:label
    private String comment = null;    // value of rdfs:comment
    private boolean implicit = false; // value of :isImplicit proeprty

    // Caches, so we don't keep allocating new Roles - key is URI
    private static volatile Map<URI,Role> cacheInternal = null;

    /** SPARQL query to get role descriptions, works on both internal repo graph and exports. */
    public static final String ROLE_QUERY =
          "SELECT DISTINCT ?uri ?label ?comment ?implicit WHERE {\n"+
          " ?uri a <"+REPO.ROLE+"> ; <"+RDFS.LABEL+"> ?label .\n"+
          "  OPTIONAL { ?uri <"+REPO.IS_IMPLICIT+"> ?implicit }\n"+
          "  OPTIONAL { ?uri <"+RDFS.COMMENT+"> ?comment }}";

    // constructor
    private Role(URI uri, String label, String comment, boolean implicit)
    {
        super();
        this.uri = uri;
        this.label = label;
        this.comment = comment;
        this.implicit = implicit;
    }

    /**
     * Get the Role for given URI, lookup label.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param uri URI of the role
     * @return a {@link org.eaglei.repository.Role} object representing role.
     * @throws javax.servlet.ServletException if any.
     */
    public static Role find(HttpServletRequest request, URI uri)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        if (cache.containsKey(uri)) {
            if (log.isDebugEnabled())
                log.debug("find("+uri.stringValue()+") => "+cache.get(uri));
            return cache.get(uri);
        } else
            throw new NotFoundException("There is no Role of URI="+uri);
    }

    /**
     * Get all known Roles - includes pseudo-Roles Anonymous and Authenticated
     * that should NOT be directly assigned to Users.
     * Make the query each time instead of relying on cache since some sneaky
     * admin may *add* new roles by uploading statements to NG_Internal graph.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<Role> findAll(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        return new ArrayList<Role>(cache.values());
    }

    /**
     * Get all Roles that can be granted in a :has___Access statement.
     * This works out to all roles *except* SuperUser.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<Role> findAllGrantable(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        List<Role> result = new ArrayList(cache.values());
        result.remove(find(request, REPO.ROLE_SUPERUSER));
        Collections.sort(result);
        return result;
    }

    /**
     * Get all Roles that ought to be assigned to users - EXCLUDES
     * "implicit" pseudo-Roles Anonymous and Authenticated.
     * Make the query each time instead of relying on cache since some sneaky
     * admin may *add* new roles by uploading statements to NG_Internal graph.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @return all roles in a {@link java.lang.Iterable} object.
     * @throws javax.servlet.ServletException if any.
     */
    public static List<Role> findAllUserRoles(HttpServletRequest request)
        throws ServletException
    {
        Map<URI,Role> cache = getCache(request);
        List<Role> result = new ArrayList<Role>();
        for (Role r : cache.values()) {
            if (!r.implicit)
                result.add(r);
        }
        return result;
    }

    /**
     * Invalidate the role cache, called when RDF has changed.
     */
    public static void decache()
    {
        synchronized (Role.class) {
            cacheInternal = null;
        }
    }

    /** any decache of an instance should decache global map too. */
    @Override
    public void decacheInstance()
    {
        decache();
    }

    // reload role cache if needed.
    private static Map<URI,Role> getCache(HttpServletRequest request)
        throws ServletException
    {
        // this looks weird but is necessary to minimize sync block.
        synchronized (Role.class) {
            if (cacheInternal != null)
                return cacheInternal;
            cacheInternal = new HashMap<URI,Role>();
        }
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, ROLE_QUERY);
            q.setDataset(SPARQL.InternalGraphs);
         
            // need inference!
            q.setIncludeInferred(true);
            SPARQL.evaluateTupleQuery(ROLE_QUERY, q, new allRoleHandler(cacheInternal));
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
        return cacheInternal;
    }

    /**
     * <p>getURI</p>
     *
     * @return URI subject of the role, a {@link org.openrdf.model.URI} object.
     */
    @Override
    public URI getURI()
    {
        return uri;
    }

    /**
     * <p>isSuperuser</p>
     * predicate that answers whether this role confers Administrator
     * privileges, an attempt at preserving modularity.
     *
     * @return true if this role determins superuser (admin) privileges
     */
    public boolean isSuperuser()
    {
        return uri.equals(REPO.ROLE_SUPERUSER);
    }

    /**
     * <p>Getter for the field <code>label</code>.</p>
     *
     * @return label of the role, a {@link java.lang.String} object.
     */
    @Override
    public String getLabel()
    {
        return label;
    }

    /**
     * <p>Getter for the field <code>comment</code>.</p>
     *
     * @return comment of the role, a {@link java.lang.String} object, MIGHT be null.
     */
    public String getComment()
    {
        return comment;
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(Object other)
    {
        return other instanceof Role && uri.equals(((Role)other).uri);
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode()
    {
        return uri.hashCode();
    }

    /**
     * <p>toString</p>
     *
     * @return a {@link java.lang.String} object.
     */
    @Override
    public String toString()
    {
        return "<#Role: uri="+uri.toString()+
            ", label="+label+">";
    }

    /** {@inheritDoc} */
    @Override
    public int compareTo(Object o)
    {
        return label.compareTo(((Role)o).label);
    }

    /** predicate - is this one of the permanent builtin roles? */
    public boolean isBuiltin()
    {
        return implicit || REPO.ROLE_SUPERUSER.equals(uri);
    }

    /** predicate - value of implicit */
    public boolean isImplicit()
    {
        return implicit;
    }

    /**
     * <p>Make a new Role</p> at a specific URI (assumes it is not
     * already the subject of any existing statements).  If uri == null,
     * a new URI is allocated.
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param label short name for role, SHOULD be non-null
     * @param comment longer description of role, may be null
     * @throws javax.servlet.ServletException if any.
     */
    public static Role create(HttpServletRequest request, URI asubject, String label, String comment, boolean aimplicit)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to create a new Role.");

        // sanity check that we aren't creating a new implicit role.
        // XXX may be allowed?
        if (aimplicit)
            throw new IllegalArgumentException("Cannot create() a Role with implicit = true.");

        // First, sanity check that the URI is not already *any* kind of object
        // Then, add role property statements.
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            if (label == null || label.length() == 0)
                throw new BadRequestException("Label is required when creating a new Role.");

            // forge a new subject URI
            URI subject = asubject;
            if (subject == null)
                subject = vf.createURI(Configuration.getInstance().getDefaultNamespace(),
                            RIDGenerator.getInstance().newID().toString());

            // rdf description of role:
            rc.add(subject, RDF.TYPE, REPO.ROLE, REPO.NG_INTERNAL);
            rc.add(subject, RDFS.LABEL, vf.createLiteral(label), REPO.NG_INTERNAL);
            if (comment != null && comment.length() > 0)
                rc.add(subject, RDFS.COMMENT, vf.createLiteral(comment), REPO.NG_INTERNAL);
            Role result = new Role(subject, label, comment, aimplicit);
            result.setDirty(true);
            decache();
            if (log.isDebugEnabled())
                log.debug("Created Role uri="+subject.stringValue()+", label="+label);
            return result;
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /** destroy a role */
    public void delete(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to delete a Role.");
        if (isBuiltin())
            throw new ForbiddenException("Built-in Roles may not be deleted.");

        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            rc.remove(uri, null, null, REPO.NG_INTERNAL);
            setDirty(true);
            decacheInstance();
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    /**
     * <p>Setter for the field <code>label</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param label a new label value
     * @throws javax.servlet.ServletException if any.
     */
    public void setLabel(HttpServletRequest request, String label)
        throws ServletException
    {
        if (label == null || label.length() == 0)
            throw new BadRequestException("Label must be a non-empty string.");
        setMetadataInternal(request, RDFS.LABEL, label);
    }

    /**
     * <p>Setter for the field <code>comment</code>.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @param newVal a new comment value
     * @throws javax.servlet.ServletException if any.
     */
    public void setComment(HttpServletRequest request, String newVal)
        throws ServletException
    {
        setMetadataInternal(request, RDFS.COMMENT, newVal);
    }


    // update one of the metadata properties.
    private void setMetadataInternal(HttpServletRequest request, URI property, String newVal)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("You must be an Administrator to modify Roles.");
        if (isBuiltin())
            throw new ForbiddenException("Builtin Roles may not be modified.");
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            ValueFactory vf = rc.getValueFactory();
            rc.remove(uri, property, null, REPO.NG_INTERNAL);
            if (newVal != null) {
                rc.add(uri, property, vf.createLiteral(newVal), REPO.NG_INTERNAL);
                log.debug("Setting Role uri="+uri+", property="+property+" to value="+newVal);
            } else {
                log.debug("Clearing Role uri="+uri+", property="+property);
            }
            setDirty(true);
        } catch (RepositoryException e) {
            throw new ServletException(e);
        }
    }

    // tuple query result handler to gather list of all roles.
    private static class allRoleHandler extends TupleQueryResultHandlerBase
    {
        private Map<URI,Role> result = null;

        allRoleHandler(Map<URI,Role> result)
        {
            super();
            allRoleHandler.this.result = result;
        }

        /** {@inheritDoc} */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            Value newURI = bs.getValue("uri");
            if (newURI instanceof URI) {
                URI u = (URI)newURI;
                    // If search is limited to "roles that can be assigned to users"
                    // skip anonymous and authenticated, they are virtual roles.
                Value rlabel = bs.getValue("label");
                String label = (rlabel instanceof Literal) ?
                           ((Literal)rlabel).getLabel() : "";
                Value rcomment = bs.getValue("comment");
                String comment = null;
                if (rcomment instanceof Literal)
                    comment = ((Literal)rcomment).getLabel();
                boolean implicit = false;
                Value rimplicit = bs.getValue("implicit");
                if (rimplicit instanceof Literal)
                    implicit = ((Literal)rimplicit).booleanValue();
                result.put(u, new Role(u, label, comment, implicit));
            } else {
                log.error("Should not get null or non-URI result in allRoleHandler: "+newURI);
            }
        }
    }
}
