package org.eaglei.repository.model;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.ServletException;

import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;

import org.eaglei.repository.util.WithRepositoryConnection;

/**
 * Outline of all *writable* "Object Model" classes, in which instances can
 * be added and changed.  Object Models are Java objects that represent an
 * instance or layer -- i.e. a body of properties on an instance -- in the
 * RDF database.  They exist to cache the information and make it easier
 * to manipulate in Java.  This class describes a common interface to
 * be implemented by all object models.
 *
 * @see ImmutableObjectModel
 * @author Larry Stone
 * Started March 2011
 */
public abstract class WritableObjectModel extends ImmutableObjectModel
{
    /**
     * flag saying the Java instance has been modified and needs to
     * be uploaded to RDF database
     */
    private boolean dirty = false;

    /**
     * Copy local changes to the RDF database.
     * Default implementation has nothing to do for cases where RDF is
     *  updated synchronously
     * @param request the HTTP request object from the servlet
     */
    public void update(HttpServletRequest request)
        throws ServletException
    {
    }

    /**
     * Invalidate local cache after changes.
     * Default implementation has nothing to do for cases where RDF is
     *  updated synchronously
     */
    public void decacheInstance()
    {
    }

    /**
     * Getter
     * @return true if dirty is set
     */
    protected boolean isDirty()
    {
        return dirty;
    }

    /**
     * Change the "dirty" flag to reflect when local change was made.
     * @param truth boolean value where true means there ARE local cahnges
     */
    protected void setDirty(boolean truth)
    {
        dirty = truth;
    }

    /**
     * <p>commit - commit any changes made to this object (and any others).</p>
     * <p>Note that this includes any other modifications made to RDF database.</p>
     *
     * @param request a {@link javax.servlet.http.HttpServletRequest} object.
     * @throws javax.servlet.ServletException if any.
     */
    public void commit(HttpServletRequest request)
        throws ServletException
    {
        try {
            if (dirty) {
                WithRepositoryConnection.get(request).commit();
                decacheInstance();
                LogManager.getLogger(this.getClass()).debug("Called commit(), committed RDF changes.");
            }
            dirty = false;
        } catch (OpenRDFException e) {
            throw new ServletException(e);
        }
    }
}
