package org.eaglei.repository.model.workflow;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.HashMap;
import java.util.Set;
import java.util.HashSet;
import java.io.IOException;
import java.io.OutputStreamWriter;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.model.URI;
import org.openrdf.model.Statement;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.repository.RepositoryResult;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;
import org.openrdf.rio.RDFHandler;

import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.AccessGrant;
import org.eaglei.repository.model.Transporter;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.util.WithRepositoryConnection;
import org.eaglei.repository.util.DuplicateArg;
import org.eaglei.repository.util.HandlerBadRequest;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.vocabulary.REPO;

/**
 * Export and import of the WorkflowTransition object.
 *
 * @author Larry Stone
 * Started March, 2011
 */
public class TransportWorkflowTransition implements Transporter
{
    private static Logger log = LogManager.getLogger(TransportWorkflowTransition.class);

    /**
     * Check that current authenticated user is authorized for this
     * operation; some export requests require Superuser privilege.
     *
     * Policy: Any authenticated user can EXPORT WFTs.
     */
    @Override
    public void authorizeExport(HttpServletRequest request)
        throws ServletException
    {
        if (Authentication.getAuthenticatedUsername(request) == null)
            throw new ForbiddenException("Export of WFTs requires an authenticated user login.");
    }

    /**
     * Check that current authenticated user is authorized for this
     * operation; some import requests require Superuser privilege.
     *
     * Policy: Only Admin user can IMPORT WTFs, since they cah
     * change the authorization situation.
     */
    @Override
    public void authorizeImport(HttpServletRequest request)
        throws ServletException
    {
        if (!Authentication.isSuperuser(request))
            throw new ForbiddenException("Import of WFTs requires administrator privileges.");
    }

    /**
     * Export transitions to serialized quad format.
     * Since a transition is simply a subject with certain required
     * and optional properties, we can manage this by dumping out all
     * statements about each relevant subject.
     * {@inheritDoc}
     */
    @Override
    public void doExport(HttpServletRequest request, HttpServletResponse response,
                       RDFFormat format, Set<String> includes, Set<String> excludes)
        throws ServletException, IOException
    {
        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);
            RDFHandler out = Rio.createWriter(format, new OutputStreamWriter(response.getOutputStream(), "UTF-8"));
            out.startRDF();
            for (WorkflowTransition t : WorkflowTransition.findAll(request)) {
                URI tu = t.getURI();
                String tus = tu.stringValue();
                String tl = t.getLabel();
                if (excludes.contains(tl) || excludes.contains(tus)) {
                    log.debug("SKIP TRANSITION because of exclude: "+t);
                } else if (includes.isEmpty() ||
                         includes.contains(tl) || includes.contains(tus)) {
                    RepositoryResult<Statement> rr = null;
                    try {
                        rr = rc.getStatements(tu, null, null, false, REPO.NG_INTERNAL);
                        while (rr.hasNext()) {
                            out.handleStatement(rr.next());
                        }
                    } finally {
                        rr.close();
                    }
                    for (Statement s : AccessGrant.exportGrants(tu, AccessGrant.getGrants(request, tu))) {
                        out.handleStatement(s);
                    }
                } else {
                    log.debug("SKIP TRANSITION because of include: "+t);
                }
            }
            out.endRDF();
        } catch (OpenRDFException e) {
            throw new InternalServerErrorException(e);
        }
    }

    /**
     * Import description of transitions from serialized RDF quads.
     * {@inheritDoc}
     */
    @Override
    public void doImport(HttpServletRequest request, HttpServletResponse response,
                       RepositoryConnection content,
                       Set<String> includes, Set<String> excludes,
                       DuplicateArg duplicate,
                       boolean transform, boolean ignoreACL)
        throws ServletException, IOException
    {
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        try {
            TupleQuery q = content.prepareTupleQuery(QueryLanguage.SPARQL, WorkflowTransition.SIMPLE_TRANSITION_QUERY);
            q.setDataset(SPARQL.InternalGraphs);
            q.setIncludeInferred(false);
            List<WorkflowTransition> result = new ArrayList<WorkflowTransition>();
            SPARQL.evaluateTupleQuery(WorkflowTransition.SIMPLE_TRANSITION_QUERY,
                                      q, new WorkflowTransition.WFTHandler(result));
            Set<URI> existingURI = new HashSet<URI>();
            Map<String,URI> existingLabel = new HashMap<String,URI>();
            for (WorkflowTransition e : WorkflowTransition.findAll(request)) {
                URI u = e.getURI();
                existingURI.add(u);
                existingLabel.put(e.getLabel(), u);
            }
            for (WorkflowTransition t : result) {
                if (log.isDebugEnabled())
                    log.debug("import Transition: Got raw result: "+t);
                URI tu = t.getURI();
                String us = tu.stringValue();
                String ls = t.getLabel();

                // first, apply include/exclude filter
                if (excludes.contains(us) || excludes.contains(ls)) {
                    log.debug("SKIP IMPORT TRANSITION because of exclude: uri="+us+", label="+ls);
                    continue;
                } else if (!includes.isEmpty() && !(includes.contains(us) || includes.contains(ls))) {
                    log.debug("SKIP IMPORT TRANSITION because of include: uri="+us+", label="+ls);
                    continue;
                }

                URI du = null;
                if (existingURI.contains(tu))
                    du = tu;
                else if (existingLabel.containsKey(ls))
                    du = existingLabel.get(ls);
                if (du != null) {
                    log.debug("Found a duplicate Transition uri="+du.stringValue());
                    if (duplicate == DuplicateArg.ignore) {
                        log.debug("SKIPPING duplicate Transition, input uri="+us);
                        continue;
                    } else if (duplicate == DuplicateArg.abort) {
                        throw new HandlerBadRequest("Import contains a duplicate Transition for existing Transition uri="+du.stringValue());
                    // to replace -- delete old transition first.
                    } else if (duplicate == DuplicateArg.replace) {
                        WorkflowTransition orig = WorkflowTransition.find(request, du);
                        orig.delete(request);
                    }
                }
                URI nu = t.write(request, transform);

                // apply access grants to the new instance if chosen.
                if (!ignoreACL) {
                    for (Statement s : AccessGrant.importGrants(request, content, tu, nu)) {
                        log.debug("Adding access grant stm: "+s);
                        rc.add(s, REPO.NG_INTERNAL);
                    }
                }
            }
        } catch (HandlerBadRequest e) {
            throw new BadRequestException(e.getMessage(), e);
        } catch (MalformedQueryException e) {
            log.error("Rejecting malformed query:"+e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }
}
