package org.eaglei.repository.servlet;

import org.eaglei.repository.util.WithRepositoryConnection;
import java.io.ByteArrayOutputStream;
import java.net.URL;
import java.net.URLEncoder;
import java.nio.charset.Charset;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.commons.mail.SimpleEmail;
import org.apache.commons.mail.Email;
import org.apache.commons.mail.EmailException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.BindingSet;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.TupleQueryResultHandlerBase;
import org.openrdf.query.TupleQueryResultHandlerException;
import org.openrdf.query.TupleQuery;
import org.openrdf.model.URI;
import org.openrdf.model.Literal;
import org.openrdf.model.Statement;
import org.openrdf.model.Value;
import org.openrdf.model.vocabulary.RDFS;
import org.openrdf.model.vocabulary.RDF;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryResult;

import org.eaglei.repository.Configuration;
import org.eaglei.repository.model.DataModel;
import org.eaglei.repository.model.View;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.NotFoundException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.util.SPARQL;

/**
 * Method: POST only.
 *
 * Query Args:
 *   uri (req) - instance about which to contact
 *   client_ip (req) - client ip addr
 *   from_name (req)
 *   from_email (req)
 *   subject
 *   message
 *   test_mode - boolean, if arg is present, redirect mail to postmaster.
 *
 * @author Larry Stone
 * @version $Id: $
 * Started June 2010
 */
public class EmailContact extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(EmailContact.class);

    // configuration property name of "postmaster" address
    private static final String CONFIG_POSTMASTER = "eaglei.repository.postmaster";
    // contact info for email
    private static final String CONFIG_MTA_HOST = "eaglei.repository.mail.host";
    private static final String CONFIG_MTA_PORT = "eaglei.repository.mail.port";
    private static final String CONFIG_MTA_SSL = "eaglei.repository.mail.ssl";
    private static final String CONFIG_MTA_USERNAME = "eaglei.repository.mail.username";
    private static final String CONFIG_MTA_PASSWORD = "eaglei.repository.mail.password";

    private static final String postmaster = Configuration.getInstance().getConfigurationProperty(CONFIG_POSTMASTER);
    private static final String mtaHost = Configuration.getInstance().getConfigurationProperty(CONFIG_MTA_HOST, "localhost");
    private static final String mtaPort = Configuration.getInstance().getConfigurationProperty(CONFIG_MTA_PORT);
    private static final String mtaUsername = Configuration.getInstance().getConfigurationProperty(CONFIG_MTA_USERNAME);
    private static final String mtaPassword = Configuration.getInstance().getConfigurationProperty(CONFIG_MTA_PASSWORD);
    private static final boolean mtaSSL = Configuration.getInstance().getConfigurationPropertyAsBoolean(CONFIG_MTA_SSL, false);

    // must bind ?subject
    private static final String emailQuery = DataModel.CONTACT_EMAIL_QUERY.getString();

    private static final String emailQueryVars[] = DataModel.CONTACT_EMAIL_BINDINGS.getArrayOfString();

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        URI uri = getParameterAsURI(request, "uri", true);
        String clientIP = getParameter(request, "client_ip", true);
        String fromName = getParameter(request, "from_name", true);
        String fromEmail = getParameter(request, "from_email", true);
        String subject = getParameter(request, "subject", false);
        String message = getParameter(request, "message", false);
        boolean testMode = isParameterPresent(request, "test_mode");

        // configuration sanity check
        if (emailQuery == null || emailQueryVars == null)
            throw new ServletException("Configuration error, email contact query is not configured correctly.");

        // *rough* sanity check, from email has something like @foo.bar at least..
        int at = fromEmail.indexOf('@');
        if (at < 0 || fromEmail.indexOf('.', at) < 0)
            throw new BadRequestException("Bad value for the required arg 'from_email', must be a valid email address.");

        // must have postmaster for bounces, default mailto
        if (postmaster == null)
            throw new InternalServerErrorException("Repository configuration is missing the required property: "+CONFIG_POSTMASTER);

        // need the referrer header to generate a link back to the JSP
        // page to report success.
        String referrer = request.getHeader("Referer");
        if (referrer == null)
            throw new BadRequestException("This service may only be called from another page (to set the referer).");
        URL backLink = new URL(referrer);
        log.debug("Got Referer = "+referrer);

        try {
            RepositoryConnection rc = WithRepositoryConnection.get(request);

            // sanity check that the resource instance exists:
            if (!rc.hasStatement(uri, RDF.TYPE, null, false)) {
                throw new NotFoundException("Subject not found in this repository: "+uri.toString());
            }

            // get rdfs:label of subject URI
            String label = null;
            RepositoryResult<Statement> rr = rc.getStatements(uri, RDFS.LABEL, null, false);
            try {
                while (rr.hasNext()) {
                    Statement s = rr.next();
                    Value rl = s.getObject();
                    if (rl instanceof Literal) {
                        label = ((Literal)rl).getLabel();
                        break;
                    }
                }
            } finally {
                rr.close();
            }

            // message content
            String myHost = request.getServerName();
            StringBuilder body = new StringBuilder();
            body.append("This message was generated by the eagle-i server on ").
                 append(myHost).append("\n");
            body.append("An eagle-i visitor, ").append(fromName).
              append(", wishes to contact the owner of this resource:\n").
              append("  Label: ").append(label == null ? "" : label).
              append("\n    URI: ").append(uri.stringValue()).append("\n");
            body.append("\nClient remote IP address = ").append(clientIP).append("\n");
            body.append("\nMessage from ").append(fromName).
                append(":\n--------------------------------------------------\n").
                append(message == null ? "(no message)" : message);
            body.append("\n--------------------------------------------------\n");

            // set up message
            Email msg = new SimpleEmail();
            msg.setCharset("UTF-8");
            msg.setMsg(body.toString());
            msg.setFrom(fromEmail, fromName);
            msg.addReplyTo(fromEmail, fromName);
            // this shows up as the envelope from address
            msg.setBounceAddress(postmaster);
            msg.setSubject(subject == null ? "Inquiry from eagle-i" : subject);
            if (testMode)
                msg.addTo(postmaster);

            // Query for email destinations:

            // Construct a dataset for objects to report on from
            // workspace or view, published resource graphs by default.
            DatasetImpl ds = new DatasetImpl();
            View.addGraphs(request, ds, View.USER_RESOURCES);

            TupleQuery q = rc.prepareTupleQuery(QueryLanguage.SPARQL, emailQuery);
            q.setDataset(ds);
            q.setIncludeInferred(false);
            q.clearBindings();
            q.setBinding("subject", uri);
            SPARQL.evaluateTupleQuery(emailQuery, q, new EmailQueryHandler(uri, msg, testMode));

            // setup MTA params
            msg.setHostName(mtaHost);
            if (mtaSSL) {
                msg.setSSL(true);
                if ((mtaUsername != null && mtaPassword == null) ||
                    (mtaUsername == null && mtaPassword != null)) {
                        throw new InternalServerErrorException("Bad configuraiton, must have BOTH username and password for mail server.");
                }
                if (mtaUsername != null &&  mtaPassword != null)
                    msg.setAuthentication(mtaUsername, mtaPassword);
                if (mtaPort != null)
                    msg.setSslSmtpPort(mtaPort);
            } else if (mtaPort != null) {
                try {
                    int p = Integer.parseInt(mtaPort);
                    msg.setSmtpPort(p);
                } catch (NumberFormatException e) {
                    throw new BadRequestException(e.toString(), e);
                }
            }

            if (log.isDebugEnabled()) {
                ByteArrayOutputStream baos = new ByteArrayOutputStream();
                msg.buildMimeMessage();
                javax.mail.internet.MimeMessage mm = msg.getMimeMessage();
                if (mm == null)
                    log.error("Failed getMimeMessage!");
                else {
                    mm.writeTo(baos);
                    log.debug("Msg = "+baos.toString());
                }
            }
            msg.send();

            String redir = backLink.getProtocol()+"://"+backLink.getHost()+
                           (backLink.getPort() < 0 ? "" : ":"+String.valueOf(backLink.getPort()))+
                           backLink.getPath()+"?sent=true&uri="+
              URLEncoder.encode(uri.stringValue(), Charset.defaultCharset().name())+
              "&test_mode="+String.valueOf(testMode);

            log.debug("Redirecting to: "+redir);
            response.sendRedirect(redir);
        } catch (javax.mail.MessagingException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (EmailException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        }
    }

    // Pick email destination(s) out of query results
    private final class EmailQueryHandler extends TupleQueryResultHandlerBase
    {
        private Email msg = null;
        private boolean testing = false;
        private URI uri = null;

        // record of index of last good value found, or -1 if none found yet.
        // it is an index into emailQueryVars
        private int lastValue = -1;

        private boolean foundSomething = false;

        private EmailQueryHandler(URI uri, Email msg, boolean testing)
        {
            super();
            EmailQueryHandler.this.uri = uri;
            EmailQueryHandler.this.msg = msg;
            EmailQueryHandler.this.testing = testing;
        }

        /**
         * Algorithm:
         * The variables named in emailQueryVars are in order of priority.
         * On first pass, or first pass with any values in the result variables,
         * pick the highest-priority variable with a value and establish that
         * as the "source" of destinations.  Walk through all result rows and
         * add any additional values of that variable to the destination
         * address list, e.g. all the "?email2" results.
         */
        @Override
        public void handleSolution(BindingSet bs)
            throws TupleQueryResultHandlerException
        {
            String dest = null;
            if (lastValue >= 0) {
                // sanity check: any higher-priority variables iwth a value?
                for (int i = 0; i < lastValue; ++i) {
                    if (bs.hasBinding(emailQueryVars[i])) {
                        log.warn("Anomaly in email results: Locked onto result var="+emailQueryVars[lastValue]+", but this higher-priority var="+emailQueryVars[i]+" has a value too in later result!  URI="+uri);
                    }
                }
                Value v = bs.getValue(emailQueryVars[lastValue]);
                if (v == null) {
                    log.debug("Got UNUSED email desination solution, no value for ?"+emailQueryVars[lastValue]);
                } else if (v instanceof Literal) {
                    dest = ((Literal)v).getLabel();
                    log.debug("Got email desination ?"+emailQueryVars[lastValue]+" = \""+dest+"\"");
                } else {
                    log.warn("Sanity check: Skipping value of email property is not a literal, uri="+uri+", value="+v);
                }
            } else {
                for (int i = 0; i < emailQueryVars.length; ++i) {
                    if (bs.hasBinding(emailQueryVars[i])) {
                        Value v = bs.getValue(emailQueryVars[i]);
                        if (v instanceof Literal) {
                            dest = ((Literal)v).getLabel();
                            lastValue = i;
                            foundSomething = true;
                            log.debug("Got email desination ?"+emailQueryVars[i]+" = \""+dest+"\"");
                            break;
                        } else {
                            log.warn("Sanity check: Skipping value of email property is not a literal, uri="+uri+", var="+emailQueryVars[i]+", value="+v);
                        }
                    }
                }
            }
            if (dest != null) {
                try {
                    if (testing) {
                        msg.addHeader("X-TESTING-Would-Have-Gone-To", dest);
                    } else {
                        msg.addTo(dest);
                    }
                } catch (EmailException e) {
                    log.error(e);
                    throw new TupleQueryResultHandlerException(e);
                }
            }
        }

        /**
         * If no destination results found, direct the mail to the postmaster.
         * Log a warning since this "shouldn't" happen, it may indicate an
         * error in the data or the query.
         */
        @Override
        public void endQueryResult()
            throws TupleQueryResultHandlerException
        {
            if (!foundSomething) {
                try {
                    msg.addTo(postmaster);
                } catch (EmailException e) {
                    log.error(e);
                    throw new TupleQueryResultHandlerException(e);
                }
                log.warn("No email contact found for resource URI="+uri+", sending to Postmaster.");
            }
        }
    }
}
