package org.eaglei.repository.servlet;

import org.eaglei.repository.util.WithRepositoryConnection;
import java.io.IOException;
import java.io.OutputStreamWriter;
import java.util.regex.Pattern;
import java.util.regex.Matcher;
import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.OpenRDFException;
import org.openrdf.repository.RepositoryConnection;
import org.openrdf.query.TupleQuery;
import org.openrdf.query.Query;
import org.openrdf.query.BooleanQuery;
import org.openrdf.query.GraphQuery;
import org.openrdf.query.QueryLanguage;
import org.openrdf.query.MalformedQueryException;
import org.openrdf.query.resultio.BooleanQueryResultFormat;
import org.openrdf.query.resultio.TupleQueryResultFormat;
import org.openrdf.query.resultio.QueryResultIO;
import org.openrdf.query.QueryInterruptedException;
import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.rio.RDFFormat;
import org.openrdf.rio.Rio;

import org.eaglei.repository.Configuration;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.model.Access;
import org.eaglei.repository.model.View;
import org.eaglei.repository.status.BadRequestException;
import org.eaglei.repository.status.InternalServerErrorException;
import org.eaglei.repository.status.ForbiddenException;
import org.eaglei.repository.status.HttpStatusException;
import org.eaglei.repository.util.Formats;
import org.eaglei.repository.util.Utils;

/**
 * SPARQL query protocol service
 * see http://www.w3.org/TR/rdf-sparql-protocol/
 *
 * Nonstandard extensions:
 *  format arg - controls the result format
 *  view - select default and named graphs from view instead of *-uri args.
 *  workspace - select default and named graphs from workspace instead of *-uri args.
 *  time - sets a time limit in seconds (NOT IMPLEMENTED)
 *  inferred - boolean param, include inferred results (default true)
 *
 * @author Larry Stone
 * @version $Id: $
 */
public class SparqlProtocol extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(SparqlProtocol.class);

    // query arg names defined in SPARQL/Protocol (others are nonstandard)
    private static final String P_QUERY = "query";
    private static final String P_NAMED_GRAPH = "named-graph-uri";
    private static final String P_DEFAULT_GRAPH = "default-graph-uri";

    private static final int MS_PER_SECOND = 1000;

    // Absolute default for max query time if nothing configured: 10 minutes.
    private static final int DEFAULT_MAX_TIME = 10 * 60;

    // configured or defaulted maximum query time, in seconds
    // see eaglei.repository.sparqlprotocol.max.time
    private static int defaultMaxQueryTime =
        Configuration.getInstance().getConfigurationPropertyAsInt("eaglei.repository.sparqlprotocol.max.time", DEFAULT_MAX_TIME);

    // Millisecond time limit after which a query is considered "slow" (ONLY controls logging)
    private static long slowQuery = 0;
    static {
        String sq = Configuration.getInstance().getConfigurationProperty("eaglei.repository.slow.query");
        try {
            if (sq != null)
                slowQuery = MS_PER_SECOND * Integer.parseInt(sq);
        } catch (NumberFormatException e) {
            log.error("Value of config property \"eaglei.repository.slow.query\" is not a legal Long integer: "+sq+": exception="+e);
        }
    }

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        this.doPost(request,response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, IOException
    {
        request.setCharacterEncoding("UTF-8");
        long startMs = System.currentTimeMillis();
        String rawquery = getParameter(request, P_QUERY, true);
        String namedGraph[] = getParameters(request, P_NAMED_GRAPH, false);
        String defaultGraph[] = getParameters(request, P_DEFAULT_GRAPH, false);
        String format = getParameter(request, "format", false);
        String rawview = getParameter(request, "view", false);
        String workspace = getParameter(request, "workspace", false);
        String rawTime = getParameter(request, "time", false);

        // non-standard arg to include (or not) inferred statements in result
        boolean inferred = getParameterAsBoolean(request, "inferred", true, false);

        // Beware: Some HTML forms can submit 0-length string for "view" etc.
        View view = null;
        if (rawview != null && rawview.length() > 0)
            view = View.parseView(rawview);

        if (workspace != null && workspace.length() == 0)
            workspace = null;

        // sanity check: workspace, view, and default/named graphs args
        // are all mutually exclusive:
        int dsCount = 0;
        if (namedGraph != null || defaultGraph != null)
            ++dsCount;
        if (workspace != null)
            ++dsCount;
        if (view != null)
            ++dsCount;
        if (dsCount > 1) {
            throw new BadRequestException("Only one of the arguments for 'view', 'workspace' or explicit named/default graph URIs maybe specified.");

        // XXX dike this if we decide to implement a default view..
        } else if (dsCount == 0) {
            throw new BadRequestException("Missing required argument: Exactly one of the arguments 'view', 'workspace' or explicit named/default graph URIs must be specified.");
        }

        // parse the SPARQL query and handle each different form (SELECT, ASK, etc)
        RepositoryConnection rc = WithRepositoryConnection.get(request);
        int time = 0;
        try {
            // set up dataset - either explicit list, or view
            // note that named graphs are needed for GRAPH keyword in SPARQL.
            DatasetImpl ds = new DatasetImpl();
            if (defaultGraph != null || namedGraph != null) {
                if (defaultGraph != null) {
                    for (String n : defaultGraph) {
                        URI gu = Utils.parseURI(n, "default-graph-uri", true);
                        if (Access.hasPermission(request, gu, Access.READ)) {
                            log.debug("Adding Default Graph to the dataset: "+n);
                            ds.addDefaultGraph(gu);
                        } else {
                            throw new ForbiddenException("Read access denied to named graph: "+gu.toString());
                        }
                    }
                }
                if (namedGraph != null) {
                    for (String n : namedGraph) {
                        URI gu = Utils.parseURI(n, "named-graph-uri", true);
                        if (Access.hasPermission(request, gu, Access.READ)) {
                            log.debug("Adding Named Graph to the dataset: "+n);
                            ds.addNamedGraph(gu);
                        } else {
                            throw new ForbiddenException("Read access denied to named graph: "+gu.toString());
                        }
                    }
                }
            } else if (view != null) {
                View.addGraphs(request, ds, view);
            } else if (workspace != null) {
                View.addWorkspaceGraphs(request, ds, Utils.parseURI(workspace, "workspace", true));
            }

            // sanity check: do not let the query proceed with empty dataset
            // because this would inadvertently give access to ALL named graphs.
            if (ds.getDefaultGraphs().isEmpty() && ds.getNamedGraphs().isEmpty()) {
                throw new InternalServerErrorException("Empty dataset for query, this should not happen.");
            }

            Query q = rc.prepareQuery(QueryLanguage.SPARQL, rawquery);
            q.setIncludeInferred(inferred);

            // For "null" view, do NOT set the explicit dataset at all.
            // This allows SPARQL's "GRAPH" operator to work, and otherwise
            // the effect is the same as an explicit dataset with a
            // graph of null -- it gives access to *all* contexts.
            if (View.NULL.equals(view))
                log.debug("Null view, leave query's Dataset unset!");

            else {
                if (log.isDebugEnabled())
                    log.debug("Dataset for SPARQL query =\n"+Utils.prettyPrint(ds, 2));
                q.setDataset(ds);
            }

            /**
             * Set the query time limit: if a value was given, it must be
             * no more than the configured default UNLESS user is an Admin.
             */
            if (rawTime != null) {
                try {
                    time = Integer.parseInt(rawTime);
                } catch (NumberFormatException e) {
                    throw new BadRequestException("Value for 'time' must be a decimal integer; this is unacceptable: "+rawTime, e);
                }
            }
            if (time > 0) {
                if (time > defaultMaxQueryTime && !Authentication.isSuperuser(request))
                    throw new ForbiddenException("You may not set a longer time limit than the default of "+defaultMaxQueryTime+" seconds.");
            } else {
                time = defaultMaxQueryTime;
            }
            q.setMaxQueryTime(time);
            log.debug("Setting query time limit to "+String.valueOf(time)+" seconds.  Default max="+String.valueOf(defaultMaxQueryTime));

            // SELECT query
            if (q instanceof TupleQuery) {
                String mimeType = Formats.negotiateTupleContent(request, format);
                TupleQueryResultFormat tqf = QueryResultIO.getWriterFormatForMIMEType(mimeType);
                if (tqf == null) {
                    log.error("Failed to get tuple query format, for mime="+mimeType);
                    throw new ServletException("Failed to get tuple query format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
                    ((TupleQuery)q).evaluate(QueryResultIO.createWriter(tqf, response.getOutputStream()));
                }

            // ASK query
            } else if (q instanceof BooleanQuery) {
                String mimeType = Formats.negotiateBooleanContent(request, format);
                BooleanQueryResultFormat bf = BooleanQueryResultFormat.forMIMEType(mimeType);
                if (bf == null) {
                    log.error("Failed to get boolean serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get boolean serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
                    QueryResultIO.createWriter(bf, response.getOutputStream()).write(((BooleanQuery)q).evaluate());
                }

            // CONSTRUCT, DESCRIBE queries
            } else if (q instanceof GraphQuery) {
                String mimeType = Formats.negotiateRDFContent(request, format);
                RDFFormat rf = Formats.getRDFOutputFormatForMIMEType(mimeType);
                if (rf == null) {
                    log.error("Failed to get RDF serialization format, for mime="+mimeType);
                    throw new ServletException("Failed to get RDF serialization format that SHOULD have been available, for mime="+mimeType);
                } else {
                    response.setContentType(Utils.makeContentType(mimeType, "UTF-8"));
                    ((GraphQuery)q).evaluate(Rio.createWriter(rf, new OutputStreamWriter(response.getOutputStream(), "UTF-8")));
                }

            } else {
                log.error("Unrecognized query type! This should never happen.");
                throw new BadRequestException("Unrecognized query type! This should never happen.");
            }
        } catch (MalformedQueryException e) {
            log.info("Rejecting malformed query.");
            throw new BadRequestException("Malformed query: "+e.toString(), e);

        // time limit was exceeded before query finished
        } catch (QueryInterruptedException e) {
            log.info("Time limit exceeded while executing SPARQL query; limit="+time+" seconds.");
            throw new HttpStatusException(HttpServletResponse.SC_REQUEST_ENTITY_TOO_LARGE,
                        "Time limit exceeded while executing SPARQL query.", e);

        // this "should never happen", means the SAIL or repo does not support generic prepareQuery()
        } catch (UnsupportedOperationException e) {
            log.error(e);
            throw new ServletException(e);
        } catch (OpenRDFException e) {
            log.error(e);
            throw new ServletException(e);
        } finally {
            // log time message with a truncated fragment of query so
            // there is some hope of matching it up against the application.
            long queryMs = System.currentTimeMillis()-startMs;
            String qtrunc = rawquery;
            Pattern qp = Pattern.compile("(select|ask|construct).*", Pattern.CASE_INSENSITIVE);
            Matcher qm = qp.matcher(rawquery);
            if (qm.find())
                qtrunc = rawquery.substring(qm.start());
            log.info("SPARQL protocol request completed in "+
                     String.format("%,d mSec. query=%.120s (...)", queryMs, qtrunc));
            if (slowQuery != 0 && queryMs > slowQuery)
                log.info(String.format("SLOW QUERY, time = %d.%03d sec, query =\n%s\n",
                         queryMs/MS_PER_SECOND, queryMs % MS_PER_SECOND, rawquery));
        }
    }
}
