package org.eaglei.repository.servlet;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.ServletException;

import org.apache.log4j.Logger;
import org.apache.log4j.LogManager;

import org.openrdf.query.impl.DatasetImpl;
import org.openrdf.model.URI;
import org.openrdf.model.vocabulary.RDFS;

import org.eaglei.repository.model.View;
import org.eaglei.repository.auth.Authentication;
import org.eaglei.repository.vocabulary.REPO;
import org.eaglei.repository.vocabulary.DCTERMS;
import org.eaglei.repository.util.SPARQL;
import org.eaglei.repository.util.Utils;
import org.eaglei.repository.status.BadRequestException;

/**
 * Returns a tabular listing of information about Resource Instances
 * and their workflow status.
 * Args:
 *  - The 'format' arg selects one of the tuple query result formats.
 *  - Optional 'workspace=URI' arg only looks in designated workspace (default
 *    is hte "user" view)
 *  - type=URI only includes instances of specified type (inferred or asserted)
 *  - owner=(self|all) - show CLAIMED instances AS WELL AS unclaimed.
 *  - detail=(brief|full) - control columns in output
 *  - addPattern=SPARQL-text - add triple pattern to query.
 *  - addResults=varname.. - add variables to the result columns.
 *
 * @author Larry Stone
 * @version $Id: $
 * Started October 20, 2010
 */
public class WorkflowResources extends RepositoryServlet
{
    private static Logger log = LogManager.getLogger(WorkflowResources.class);

    // values of 'owner' arg - class must be public so utils can see it.
    public enum OwnerArg { self, all, none };

    // values of 'detail' arg - class must be public so utils can see it.
    public enum DetailArg { brief, full };

    /** {@inheritDoc} */
    @Override
    protected void doGet(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        doPost(request, response);
    }

    /** {@inheritDoc} */
    @Override
    protected void doPost(HttpServletRequest request, HttpServletResponse response)
        throws ServletException, java.io.IOException
    {
        request.setCharacterEncoding("UTF-8");
        String rawState = getParameter(request, "state", false);
        URI type = getParameterAsURI(request, "type", false);
        OwnerArg owner = (OwnerArg)getParameterAsKeyword(request, "owner", OwnerArg.class, OwnerArg.self, false);
        boolean unclaimed = getParameterAsBoolean(request, "unclaimed", true, false);
        URI workspace = getParameterAsURI(request, "workspace", false);
        String format = getParameter(request, "format", false);
        DetailArg detail = (DetailArg)getParameterAsKeyword(request, "detail", DetailArg.class, DetailArg.full, false);
        String addPattern = getParameter(request, "addPattern", false);
        String addResults = getParameter(request, "addResults", false);
        String addModifiers = getParameter(request, "addModifiers", false);

        // state is an annoyingly special case since it can also be 'all', oops.
        URI state = null;
        if (rawState == null || rawState.equals("all"))
            state = null;
        else
            state = Utils.parseURI(rawState, "state", false);

        StringBuilder qb = new StringBuilder();
        qb.append("SELECT ?r_subject ?r_label ?r_type ");
        if (detail == DetailArg.full)
            qb.append("?r_created ?r_owner ?r_ownerLabel ?r_state ");
        if (addResults != null)
            qb.append(addResults);
        qb.append(" WHERE {\n" /*}*/ );
        // requiring :hasWorkflowState will select only instances, they ALL should have it
        qb.append("  ?r_subject <").append(REPO.HAS_WORKFLOW_STATE).append("> ?r_state .\n");
        if (state != null)
            qb.append("  FILTER (?r_state = <").append(state).append(">)\n");
        // get the asserted type - but we can't turn off inferencing since
        // it's needed for the optional type matching.
        qb.append(" GRAPH ?atg {?r_subject a ?r_type }\n")
          .append(" FILTER (?atg != <").append(REPO.NG_INFERRED).append(">)\n");
        // filter by asserted-OR-inferred tyep:
        if (type != null)
            qb.append(" ?r_subject a <").append(type).append("> .\n");
        // rdfs:label
        qb.append("  OPTIONAL{ ?r_subject <" /*}*/ ).append(RDFS.LABEL).append(/*{*/ "> ?r_label}\n");
        // DC "created" date - in NG_Metadata
        qb.append("  OPTIONAL{ ?r_subject <" /*}*/ ).append(DCTERMS.CREATED).append(/*{*/ "> ?r_created }\n");
        // workflow owner - in NG_Metadata
        qb.append("  OPTIONAL{ ?r_subject <" /*}*/ ).append(REPO.HAS_WORKFLOW_OWNER).append("> ?r_owner .\n");
        qb.append("    OPTIONAL{ ?r_owner <" /*}*/ ).append(RDFS.LABEL).append(/*{{*/ "> ?r_ownerLabel }} \n");

        // filter is controlled by 'owner' and 'unclaimed' options:
        if (unclaimed && owner != OwnerArg.all) {
            qb.append("  FILTER (!BOUND(?r_owner)" /*)*/);
            if (owner == OwnerArg.self)
                qb.append(" || ?r_owner = <").append(Authentication.getPrincipalURI(request)).append(">\n");
            qb.append(/*(*/")\n");
        } else if (!unclaimed) {
            qb.append("  FILTER (" /*)*/);
            if (owner == OwnerArg.self)
                qb.append("?r_owner = <").append(Authentication.getPrincipalURI(request)).append(">\n");
            else if (owner == OwnerArg.all)
                qb.append("BOUND(?r_owner)");
            else
                throw new BadRequestException("Illegal combination of options: specifying unclaimed=false and owner=none would never produce any results.");
            qb.append(/*(*/")\n");
        }

        if (addPattern != null)
          qb.append(addPattern);
        qb.append(/*{*/ "}");
        if (addModifiers != null)
          qb.append(" ").append(addModifiers);
        String query = qb.toString();

        // Construct a dataset for objects to report on from
        // workspace if specified, or default to user-resources view
        DatasetImpl ds = new DatasetImpl();
        if (workspace != null) {
            View.addWorkspaceGraphs(request, ds, workspace);
        } else {
            View.addGraphs(request, ds, View.USER);
        }
        SPARQL.tupleQueryRequest(request, response, query, format, ds);
    }
}
