package org.eaglei.search.provider;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.EIURI;

/**
 * Represents a search request.
 */
public class SearchRequest implements Serializable {

    public static final long serialVersionUID = 1L;

    /**
     * Represents the search focus: a search query (w/ advanced syntax) and/or an 
     * explicit entity URI. If both are specified, they are combined using an implicit OR. 
     * 
     * If an entity URI is specified, it is used to retrieve resources:
     * <ul> 
     * <li>whose textual description matches the textual description of the entity
     * <li>that are instances of the entity (if the entity is a class)
     * <li>that are related to the entity via resource properties
     * </ul>
     */
    // TODO split URI and query into separate QueryTerm and EntityTerm classes;
    //      allow multiple to be specified with a boolean occurrence indicator for each
    public static class Term implements Serializable {

        public static final long serialVersionUID = 1L;

        /**
         * Optional field that holds the value of the search query string.
         * Either query or entity must be specified. May contain advanced syntax.
         */
        private String query;

        /**
         * Optional field that holds the URI an entity (may be a class or instance)
         * that is the focus of the search.
         * from the eagle-i ontology used to constrain the search term. Either
         * query or entity must be specified.
         */
        private EIURI uri;

        /*
         * 
         */
        private Term() {
            // for GWT
        }

        /**
         * Creates a new search term with the specified query string and entity
         * URI. At least one of these must be non-null. If both are specified,
         * they are combined using an implicit OR.
         * 
         * @param query Query used for a full-text searching. Must not be a zero
         *            length string.
         * @param uri The URI of an entity (class or instance) 
         *            that is used as the target for the search
         *            operation. 
         */
        public Term(final String query, final EIURI uri) {
            assert (query != null || uri != null);
            assert (query == null || query.length() > 0);
            this.query = query;
            this.uri = uri;
        }

        /**
         * Creates a new search term with the specified query string.
         * 
         * @param query Query used for a full-text searching. 
         */
        public Term(final String query) {
            this(query, null);
        }

        /**
         * Creates a new search term with the specified entity URI.
         * 
         * @param uri The URI of a class or instance that is used as the target 
         *            for the search operation.
         */
        public Term(final EIURI uri) {
            this(null, uri);
        }

        /**
         * Creates a deep copy of the given Term.
         * 
         * @param term Term to copy
         */
        public Term(final Term term) {
            this(term.query, term.uri);
        }

        /**
         * Gets the search query
         * 
         * @return The search query. May be null.
         */
        public String getQuery() {
            return this.query;
        }

        /**
         * Get the URI entity used to constrain the search.
         * 
         * @return Entity URI. May be null.
         */
        public EIURI getURI() {
            return this.uri;
        }

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
			result = prime * result + ((query == null) ? 0 : query.hashCode());
			result = prime * result + ((uri == null) ? 0 : uri.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			Term other = (Term) obj;
			if (query == null) {
				if (other.query != null) {
					return false;
				}
			} else if (!query.equals(other.query)) {
				return false;
			}
			if (uri == null) {
				if (other.uri != null) {
					return false;
				}
			} else if (!uri.equals(other.uri)) {
				return false;
			}
			return true;
		}
    }

    /**
     * Set of resource type-specific bindings for a search. These bindings are
     * specified in reference to the eagle-i ontology. NOT multi-thread safe.
     */
    public static class TypeBinding implements Serializable {

        public static final long serialVersionUID = 1L;

        /*
         * Type for search result resources. Will be the URI of a class in the
         * eagle-i ontology.
         */
        private EIURI type;

        /*
         * Constraints for data type properties for the specified type.
         */
        private Map<EIURI, String> datatypePropertyConstraints;
        
        /*
         * Constraints for the object properties of the specified type.
         */
        private Map<EIURI, EIURI> objectPropertyConstraints;

        private TypeBinding() {
            // GWT
        }

        /**
         * Creates a new TypeBinding object for the specified eagle-i class.
         * 
         * @param type A type from the eagle-i ontology.
         */
        public TypeBinding(EIURI type) {
            setType(type);
        }

        /**
         * Retrieves the type constraint for search result resources. A URI of a
         * class in the eagle-i ontology.
         * 
         * @return Type constraint.
         */
        public EIURI getType() {
            return this.type;
        }
        
        /**
         * Sets the type constraint for search result resources.
         * @param type Type constraint
         */
        public void setType(EIURI type) {
            assert type != null;
            this.type = type;            
        }
        
        /**
         * Returns the URIs (as EIURI instances) for the result data type properties.
         * 
         * @return Set of EIURIs representing a eagle-i ontology data type properties for the
         *         result.
         */
        @SuppressWarnings("unchecked")
        public Set<EIURI> getDataTypeProperties() {
            if (this.datatypePropertyConstraints == null) {
                // GWT compiler doesn't like emptySet()
                // return Collections.emptySet();
                return Collections.EMPTY_SET;
            }
            return this.datatypePropertyConstraints.keySet();
        }

        /**
         * Adds a property binding for a data type property.
         * 
         * @param property Property URI as an EIURI. Cannot be null.
         * @param value Property value.
         */
        public void addDataTypeProperty(EIURI property, String value) {
            assert property != null;
            assert value != null;
            // Not threadsafe
            if (this.datatypePropertyConstraints == null) {
                this.datatypePropertyConstraints = new HashMap<EIURI, String>();
            }
            
            this.datatypePropertyConstraints.put(property, value);
        }

        /**
         * Retrieves the constraint of the specified data type property.
         * 
         * @param property Property URI as an EIURI.
         * @return Constraint of the property if it has been set or null.
         */
        public String getDataTypeProperty(EIURI property) {
            assert property != null;
            if (this.datatypePropertyConstraints == null) {
                return null;
            }
            return this.datatypePropertyConstraints.get(property);
        }
        
        /**
         * Returns the URIs (as EIURI instances) for the result object properties.
         * 
         * @return Set of EIURIs representing a eagle-i ontology properties for the
         *         result.
         */
        @SuppressWarnings("unchecked")
        public Set<EIURI> getObjectProperties() {
            if (this.objectPropertyConstraints== null) {
                // GWT compiler doesn't like emptySet()
                // return Collections.emptySet();
                return Collections.EMPTY_SET;
            }
            return this.objectPropertyConstraints.keySet();
        }

        /**
         * Adds a property binding for an object property.
         * 
         * @param property Property URI as an EIURI. Cannot be null.
         * @param value Property value.
         */
        public void addObjectProperty(EIURI property, EIURI value) {
            assert property != null;
            assert value != null;
            // Not threadsafe
            if (this.objectPropertyConstraints == null) {
                this.objectPropertyConstraints= new HashMap<EIURI, EIURI>();
            }
            this.objectPropertyConstraints.put(property, value);
        }

        /**
         * Retrieves the constraints for the specified object property.
         * 
         * @param property Property URI as an EIURI.
         * @return Values of the property if it has been set or null.
         */
        public EIURI getObjectProperty(EIURI property) {
            assert property != null;
            if (this.objectPropertyConstraints == null) {
                return null;
            }
            return this.objectPropertyConstraints.get(property);
        }

		@Override
		public int hashCode() {
			final int prime = 31;
			int result = 1;
            result = prime * result + ((datatypePropertyConstraints == null) ? 0 : datatypePropertyConstraints.hashCode());
            result = prime * result + ((objectPropertyConstraints == null) ? 0 : objectPropertyConstraints.hashCode());            
			result = prime * result + ((type == null) ? 0 : type.hashCode());
			return result;
		}

		@Override
		public boolean equals(Object obj) {
			if (this == obj) {
				return true;
			}
			if (obj == null) {
				return false;
			}
			if (getClass() != obj.getClass()) {
				return false;
			}
			TypeBinding other = (TypeBinding) obj;
		    if (datatypePropertyConstraints == null) {
	            if (other.datatypePropertyConstraints != null) {
	                return false;
	            }
	        } else if (!datatypePropertyConstraints.equals(other.datatypePropertyConstraints)) {
	            return false;
	        }
	        if (objectPropertyConstraints == null) {
	            if (other.objectPropertyConstraints != null) {
	                return false;
	            }
	        } else if (!objectPropertyConstraints.equals(other.objectPropertyConstraints)) {
	            return false;
	        }       			
			if (type == null) {
				if (other.type != null) {
					return false;
				}
			} else if (!type.equals(other.type)) {
				return false;
			}
			return true;
		}
    }

    public static final int DEFAULT_PAGE_SIZE = 10;

    private static final String PARAM_DELIMITER = "&";
    private static final String QUERY_KEY = "q";
    private static final String URI_KEY = "uri";
    private static final String INSTITUTION_KEY = "inst";
    private static final String TYPE_KEY = "t";
    private static final String DATATYPE_PROP_KEY = "dt_";
    private static final String OBJECT_PROP_KEY = "ob_";    
    private static final String START_KEY = "start";
    private static final String MAX_KEY = "max";

    /*
     * Optional. URI of the eagle-i institution instance used to constrain the
     * search.
     */
    private EIURI institution;
    // TODO update to support multiple institutions    
    //private List<EIURI> institutions;

    /*
     * Optional. Search term. Contains a query string and/or entity
     */
    private Term term;
    // TODO update to support multiple Terms
    //private List<Term> terms;    

    /*
     * Optional. Constrain results to the given type and property bindings.
     */
    private TypeBinding binding;
    // TODO support boolean combinations of TypeBindings
    //private List<TypeBinding> bindings;

    // TODO use a common term base class for all constraints that includes boolean 
    //      occurrence info
    // TODO potentially support property constraints independent from a type
    
    /*
     * Start index, 0-based
     */
    private int startIndex = 0;

    /*
     * Max results.
     */
    private int maxResults = DEFAULT_PAGE_SIZE;

    /**
     * Creates a new "everything" SearchRequest 
     */
    public SearchRequest() {
    }

    /**
     * Creates a new SearchRequest for the specified term
     */
    public SearchRequest(Term term) {
        this.term = term;
    }

    /**
     * Parses a url param list into a search request. Invalid params will be
     * ignored. 
     * 
     * @param urlParams
     * @return SearchRequest
     */
    public SearchRequest(String urlParams) {
        if (urlParams != null && urlParams.length() > 0) {
            String[] params = urlParams.split(PARAM_DELIMITER);
            for (String param : params) {
                String[] parts = param.split("=");
                if (parts.length != 2) {
                    // ignore
                    continue;
                    // throw new Exception("Invalid param: " + param); // TODO
                    // offset...
                }
                if (parts[0].equals(QUERY_KEY)) {
                    if (term == null) {
                        term = new Term();
                    }
                    // validate?
                    term.query = parts[1];
                } else if (parts[0].equals(URI_KEY)) {
                    if (term == null) {
                        term = new Term();
                    }
                    // For now, uri string MUST omit the prefix.
                    EIURI uri = EIURI.create(parts[1]);
                    term.uri = uri;
                } else if (parts[0].equals(INSTITUTION_KEY)) {
                    EIURI uri = EIURI.create(parts[1]);
                    institution = uri;
                } else if (parts[0].equals(TYPE_KEY)) {
                    if (binding == null) {
                        binding = new TypeBinding();
                    }
                    // Support serializing the type segment of the URI
                    // TODO Factor out application-specific type lookup
                    // for now, just create a new URI
                    EIURI uri = EIURI.create(parts[1]);
                    binding.type = uri;
                } else if (parts[0].equals(START_KEY)) {
                    try {
                        setStartIndex(Integer.valueOf(parts[1]));
                    } catch (NumberFormatException ex) {
                        // ignore
                        // throw new Exception("Invalid page param: " +
                        // ex.getMessage()); // TODO
                        // offset...
                    }
                } else if (parts[0].equals(MAX_KEY)) {
                    try {
                        setMaxResults(Integer.valueOf(parts[1]));
                    } catch (NumberFormatException ex) {
                        // ignore
                        // throw new Exception("Invalid page size param: " +
                        // ex.getMessage()); // TODO
                        // offset...
                    }
                } else if (parts[0].startsWith(DATATYPE_PROP_KEY)) {
                    // binding property
                    if (binding == null) {
                        binding = new TypeBinding();
                    }
                    final String prop = parts[0].substring(DATATYPE_PROP_KEY.length());
                    EIURI uri = EIURI.create(prop);
                    binding.addDataTypeProperty(uri, parts[1]);
                } else if (parts[0].startsWith(OBJECT_PROP_KEY)) {
                    // binding property
                    if (binding == null) {
                        binding = new TypeBinding();
                    }
                    final String prop = parts[0].substring(OBJECT_PROP_KEY.length());
                    EIURI propURI = EIURI.create(prop);
                    EIURI valueURI = EIURI.create(parts[1]);
                    binding.addObjectProperty(propURI, valueURI); 
                }
            }
        }
    }

    /**
     * Get the URI for the eagle-i institution instance used to constrain the
     * search term
     * 
     * @return Institution URI. May be null.
     */
    public EIURI getInstitution() {
        return this.institution;
    }

    /**
     * Sets the URI for the eagle-i institution instance used to constrain the
     * search term.
     * 
     * @param institution Institution URI.
     */
    public void setInstitution(final EIURI institution) {
        this.institution = institution;
    }

    /**
     * Gets the search term, may be null.
     */
    public Term getTerm() {
        return this.term;
    }

    /**
     * Sets the search term.
     * 
     * @param term The search term
     */
    public void setTerm(final Term term) {
        this.term = term;
    }

    /**
     * Gets the search resource type binding, may be null.
     */
    public TypeBinding getBinding() {
        return this.binding;
    }

    /**
     * Sets the search resource type binding.
     * 
     * @param binding The search result resource type binding.
     */
    public void setBinding(final TypeBinding binding) {
        this.binding = binding;
    }

    /**
     * Gets the request start index.
     * 
     * @return The request start index.
     */
    public int getStartIndex() {
        return this.startIndex;
    }

    /**
     * Sets the search request start index.
     * 
     * @param startIndex The start index.
     */
    public void setStartIndex(final int startIndex) {
        assert startIndex >= 0;
        this.startIndex = startIndex;
    }

    /**
     * Gets the max number of results.
     * 
     * @return Maximum results to return.
     */
    public int getMaxResults() {
        return this.maxResults;
    }

    /**
     * Sets the maximum results to return.
     * 
     * @param maxResults Maximum number of results to return.
     */
    public void setMaxResults(final int maxResults) {
        assert maxResults >= 0;
        this.maxResults = maxResults;
    }

    /**
     * Returns a string representation of this request suitable for use as the
     * param list in a catalyst search URL.
     * 
     * @return param String
     */
    public String toURLParams() {
        StringBuilder buf = new StringBuilder();
        if (term != null) {
            if (term.query != null && term.query.length() > 0) {
                buf.append(QUERY_KEY);
                buf.append("=");
                buf.append(term.query);
            }
            if (term.uri != null) {
                if (buf.length() > 0) {
                    buf.append(PARAM_DELIMITER);
                }
                buf.append(URI_KEY);
                buf.append("=");
                buf.append(term.uri.toString());
            }
        }
        if (institution != null) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(INSTITUTION_KEY);
            buf.append("=");
            buf.append(institution.toString());
        }
        // Don't bother writing a page param if on page 1.
        if (startIndex > 0) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(START_KEY);
            buf.append("=");
            buf.append(startIndex);
        }
        if (maxResults != DEFAULT_PAGE_SIZE) {
            if (buf.length() > 0) {
                buf.append(PARAM_DELIMITER);
            }
            buf.append(MAX_KEY);
            buf.append("=");
            buf.append(maxResults);
        }
        if (binding != null) {
            if (binding.type != null) {
                if (buf.length() > 0) {
                    buf.append(PARAM_DELIMITER);
                }
                buf.append(TYPE_KEY);
                buf.append("=");
                // Support serializing just the type segment of the URI
                buf.append(binding.type.toString());
            }
            for (EIURI uri: binding.getDataTypeProperties()) {
                buf.append(PARAM_DELIMITER); 
                buf.append(DATATYPE_PROP_KEY);                 
                buf.append(uri.toString()); 
                buf.append("=");
                buf.append(binding.getDataTypeProperty(uri));
            }
            for (EIURI uri: binding.getObjectProperties()) {
                buf.append(PARAM_DELIMITER);
                buf.append(OBJECT_PROP_KEY);                                 
                buf.append(uri.toString()); 
                buf.append("=");
                buf.append(binding.getObjectProperty(uri));                
            }
        }

        return buf.toString();
    }

    @Override
    public String toString() {
        return toURLParams();
    }

	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((binding == null) ? 0 : binding.hashCode());
		result = prime * result + ((institution == null) ? 0 : institution.hashCode());
		result = prime * result + maxResults;
		result = prime * result + startIndex;
		result = prime * result + ((term == null) ? 0 : term.hashCode());
		return result;
	}

	public boolean equals(Object obj) {
	    return equals(obj, false);
	}
	
	/**
	 * Checks equality, optionally ignoring difference in the start index.
	 * Useful for checking if this is a page request.
	 * 
	 * @param obj
	 * @param exceptStartIndex
	 * @return
	 */
    public boolean equals(Object obj, boolean exceptStartIndex) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		SearchRequest other = (SearchRequest) obj;
		if (binding == null) {
			if (other.binding != null) {
				return false;
			}
		} else if (!binding.equals(other.binding)) {
			return false;
		}
		if (institution == null) {
			if (other.institution != null) {
				return false;
			}
		} else if (!institution.equals(other.institution)) {
			return false;
		}
		if (maxResults != other.maxResults) {
			return false;
		}
		if (!exceptStartIndex && startIndex != other.startIndex) {
			return false;
		}
		if (term == null) {
			if (other.term != null) {
				return false;
			}
		} else if (!term.equals(other.term)) {
			return false;
		}
		return true;
	}
}
