package org.eaglei.search.provider;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.EIURI;

/**
 * Holds the number of results for a specific search for a set of eagle-i ontology
 * classes. NOT thread safe.
 */
public class SearchCounts implements Serializable {

    public static final long serialVersionUID = 1L;

    /*
     * The request
     */
    private SearchRequest request;    
    
    /*
     * Map from class EIURI to a count of entities of that class which match the
     * search request.
     */
    private Map<EIURI, Integer> classToCount;

    private SearchCounts() {
        // for gwt
    }
    
    /**
     * Creates a new SearchCounts
     * 
     * @param request The request
     */
    public SearchCounts(final SearchRequest request) {
        this.request = request;
    }

    /**
     * Gets the search request.
     * 
     * @return Search request.
     */
    public SearchRequest getRequest() {
        return this.request;
    }
    
    /**
     * Returns the URIs (as EIURI instances) of the eagle-i classes for which search
     * result counts have been computed.
     * 
     * @return Set of EIURIs representing a eagle-i ontology classes for which 
     *         search result counts have been computed.
     */
    @SuppressWarnings("unchecked")
    public Set<EIURI> getClassesForCounts() {
        if (this.classToCount == null) {
            // GWT compiler doesn't like emptySet()
            // return Collections.emptySet();
            return Collections.EMPTY_SET;
        }
        return this.classToCount.keySet();
    }

    /**
     * Sets the search result count for the specified class.
     * 
     * @param classURI URI of a class in the eagle-i ontology.
     * @param count Number of results for the associated search request. 
     */
    public void setClassCount(EIURI classURI, int count) {
        // Not threadsafe
        if (this.classToCount == null) {
            this.classToCount = new HashMap<EIURI, Integer>();
        }
        this.classToCount.put(classURI, Integer.valueOf(count));
    }

    /**
     * Retrieves the number of search results for the specified class URI.
     * 
     * @param classURI URI of the ontology class whose result count is being retrieved.
     * 
     * @return Number of results of the associated search for the specified class.
     */
    public int getClassCount(EIURI classURI) {
        if (this.classToCount == null|| !this.classToCount.containsKey(classURI)) {
            return 0;
        }
        return this.classToCount.get(classURI).intValue();
    }
    
    @Override
    public boolean equals(Object o) {
        if (!(o instanceof SearchCounts)) {
            return false;
        }
        final SearchCounts other = (SearchCounts) o;
        if (request == null) {
            if (other.request != null) {
                return false;
            }
        } else if (!request.equals(other.request)) {
            return false;
        }        
        if (classToCount == null) {
            if (other.classToCount != null) {
                return false;
            }
        } else if (!classToCount.equals(other.classToCount)) {
            return false;
        }        
        return true;
    }
    
    @Override
    public int hashCode() {
        final int prime = 31;
        int result = 1;
        result = prime * result + ((request == null) ? 0 : request.hashCode());        
        result = prime * result + ((this.classToCount == null) ? 0 : this.classToCount.hashCode()); 
        return result;        
    }
    
    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder();
        
        if (this.classToCount == null) {
        	sb.append("null");
        } else {
	        for (EIURI eiuri: this.classToCount.keySet()) {
	            if (eiuri == null) {
	                sb.append("null");
	            } else {
	                sb.append(eiuri.toString());
	            }
	            sb.append(": ");
	            sb.append(this.classToCount.get(eiuri).toString());
	            sb.append("\n");
	        }
        }
        return sb.toString();
    }
    
}
