package org.eaglei.search.provider;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.List;

import org.eaglei.model.EIURI;
import org.eaglei.search.events.ChangeEventPayload;
import org.eaglei.search.events.ChangeEventPayloadItem;

/**
 * Set of search results.
 */
public class SearchResultSet implements Serializable {

    public static final long serialVersionUID = 1L;

    /*
     * The request
     */
    private SearchRequest request;

    /*
     * Total number of results
     */
    private int totalCount;

    /*
     * Start index for these results. 0-based.
     */
    private int startIndex;

    /*
     * Portion of results that are being returned
     */
    private List<SearchResult> results = new ArrayList<SearchResult>();
    
    /* 
     * True if timeout occurred, and result set is partial. 
     */
    private boolean wasTimeout = false;

    /* 
     * If partial results, list of institutional data included 
     * Undefined if no timeout occurred.
     */
    private List<EIURI> institutionsIncluded;

    /*
     * 
     */
    private SearchResultSet() {
        // for gwt
    }

    /**
     * Creates a new SearchResultSet
     * 
     * @param request The request
     */
    public SearchResultSet(final SearchRequest request) {
        this.request = request;
    }

    /** 
     * True if timeout occurred, and result set is partial. 
     */
    public boolean wasTimeout() {
        return wasTimeout;
    }

    public void setWasTimeout(boolean wasTimeout) {
        this.wasTimeout = wasTimeout;
    }

    /** 
     * If partial results, list of institutional data included 
     * Undefined if no timeout occurred.
     */
    public List<EIURI> getInstitutionsIncluded() {
        return institutionsIncluded;
    }

    public void setInstitutionsIncluded(List<EIURI> institutionsIncluded) {
        this.institutionsIncluded = institutionsIncluded;
    }

    /**
     * Gets the search request.
     * 
     * @return Search request.
     */
    public SearchRequest getRequest() {
        return this.request;
    }

    /**
     * Gets all of the SearchResults
     */
    public List<SearchResult> getResults() {
        return this.results;
    }
    
    /**
     * Total number of results for the search. Only a subset may be included in
     * this result set.
     * 
     * @return Total number of results
     */
    public int getTotalCount() {
        return this.totalCount;
    }

    /**
     * Sets the total number of results for the search.
     * 
     * @param totalCount Total number of results.
     */
    public void setTotalCount(final int totalCount) {
        this.totalCount = totalCount;
    }

    /**
     * Start index for this result set. 0-based.
     * 
     * @return 1-based start index.
     */
    public int getStartIndex() {
        return this.startIndex;
    }

    /**
     * Sets the start index.
     * 
     * @param startIndex 0-based start index
     */
    public void setStartIndex(final int startIndex) {
        assert startIndex >= 0;
        this.startIndex = startIndex;
    }

	@Override
	public int hashCode() {
		final int prime = 31;
		int result = 1;
		result = prime * result + ((request == null) ? 0 : request.hashCode());
		result = prime * result + ((results == null) ? 0 : results.hashCode());
		result = prime * result + startIndex;
		result = prime * result + totalCount;
		return result;
	}

	@Override
	public boolean equals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		SearchResultSet other = (SearchResultSet) obj;
		if (request == null) {
			if (other.request != null) {
				return false;
			}
		} else if (!request.equals(other.request)) {
			return false;
		}
		if (results == null) {
			if (other.results != null) {
				return false;
			}
		} else if (!results.equals(other.results)) {
			return false;
		}
		if (startIndex != other.startIndex) {
			return false;
		}
		if (totalCount != other.totalCount) {
			return false;
		}
		return true;
	}
	
	//NB: For serialization tests only.  We need a way to test all the fields in one SearchResultSet,
	//especially member SearchResults. Ensures that comparisons on SearchResults use SearchResult.deepEquals().
	//-Clint
	public boolean deepEquals(Object obj) {
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		SearchResultSet other = (SearchResultSet) obj;
		if (request == null) {
			if (other.request != null) {
				return false;
			}
		} else if (!request.equals(other.request)) {
			return false;
		}
		if (results == null) {
			if (other.results != null) {
				return false;
			}
		} else if(other.results == null) {
			return false;
		}
		else {
			if(results.size() != other.results.size()) {
				return false;
			}
			
			for(int i = 0; i < results.size(); ++i) {
				final SearchResult mine = results.get(0);
				final SearchResult theirs = other.results.get(0);
				
				if (mine == null) {
					if (theirs != null) {
						return false;
					}
				} else if (!mine.deepEquals(theirs)) {
					return false;
				}
			}
		}
		if (startIndex != other.startIndex) {
			return false;
		}
		if (totalCount != other.totalCount) {
			return false;
		}
		return true;
	}
}
