package org.eaglei.search.datagen;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EagleIOntConstants;
import org.eaglei.model.jena.EagleIOntDocumentManager;
import org.eaglei.model.jena.EagleIOntUtils;

import com.hp.hpl.jena.ontology.OntClass;
import com.hp.hpl.jena.ontology.OntModel;
import com.hp.hpl.jena.util.iterator.ExtendedIterator;
import com.hp.hpl.jena.vocabulary.OWL;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * Abstract base class for logic that auto-generating fake data based on the
 * eagle-i ontology.
 */
public abstract class AbstractGenerator {

    private static final Log logger = LogFactory.getLog(AbstractGenerator.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    
    public AbstractGenerator() {
    }

    /**
     * Generates for each institution specified in the params.
     * 
     * @param params
     * @throws IOException
     */
    public void generate(final DataGenParams params) throws IOException {
        if (DEBUG) {
            logger.debug("Starting generation...");
        }
        initializeStorage();
        int total = 0;
        for (String uri : params.getInstitutionURIs()) {
            String label = params.getInstitutionName(uri);
            int numGenerated = generateForInstitution(uri, label, params);
            total += numGenerated;
            logger.info("Generated " + numGenerated + " resources for " + label);            
            if (DEBUG) {
                logger.debug("Generated " + numGenerated + " resources for " + label);
            }
        }
        if (DEBUG) {
            logger.debug("Generated " + total + " resources.");
        }
        closeStorage();
        if (DEBUG) {
            logger.debug("...generation finished.");
        }
    }

    /**
     * Initializes the generator-specific storage.
     * 
     * @throws IOException
     */
    public abstract void initializeStorage() throws IOException;

    /**
     * Closes the generator-specific storage.
     * 
     * @throws IOException
     */
    public abstract void closeStorage() throws IOException;

    /**
     * Populates the Lucene index for a single university.
     * 
     * @param uri The institution URI
     * @param label The institution label
     * @param genParams Parameters that controls how the data for the
     *            institution is generated.
     * @param writer The index writer for the directory.
     * 
     * @return Total numb
     * 
     * @throws IOException Thrown if an error is encountered during data gen.
     */
    public int generateForInstitution(final String uri, final String label,
            final DataGenParams genParams) throws IOException {

        int total = 0;
        // create an arbitrary number of research facilities
        final int numCores = genParams.getNumCoreFacilities();
        if (DEBUG) {
            logger.debug("Generating " + numCores + " core facilities for " + label + ": " + uri);
        }
        for (int i = 1; i <= numCores; i++) {
            total += generateLab(uri, label, getCoreFacilityName(label, i, genParams), getCoreFacilityURI(
                    uri, i, genParams), null, EagleIOntConstants.CORE_FACILITY_CLASS_URI, genParams);
        }

        // create an arbitrary number of research facilities
        final int numResearch = genParams.getNumResearchFacilities();
        if (DEBUG) {
            logger.debug("Generating " + numResearch + " research facilities for " + label + ": " + uri);
        }
        for (int i = 1; i <= numResearch; i++) {
            total += generateLab(uri, label, getResearchFacilityName(label, i, genParams),
                    getResearchFacilityURI(uri, i, genParams), null,
                    EagleIOntConstants.RESEARCH_FACILITY_CLASS_URI, genParams);
        }
        return total;
    }

    /**
     * Generates the test RDF for a single research facility
     * 
     * @param label Label for the lab resource
     * @param labURI URI for the lab
     * @param labTypeURI URI for the lab subclass
     * @param labNumber Number of the lab.
     * @param university RDF resource that represents the university.
     * @param genParams Parameters that controls how the data for the lab is
     *            generated.
     * 
     * @return Total number of resources generated.
     * 
     * @throws IOException Thrown if an error is encountered.
     */
    public int generateLab(final String institutionURI, final String institutionLabel,
            final String labLabel, final String labURI, final String labURL, final String labTypeURI,
            final DataGenParams genParams) throws IOException {
        // create an arbitrary number of resources for each top-level class
        int totalCount = 0;
        List<OntClass> topLevelClasses = EagleIOntUtils.getClassesInGroup(EagleIOntConstants.TOP_LEVEL_GROUP);
        for (OntClass topLevelClass : topLevelClasses) {
            
            // skip the organization top-level class
            if (topLevelClass.getURI().equals(EagleIOntConstants.ORGANIZATION_CLASS_URI)) {
                continue;
            }
            
            // get a random subclass (full set of subclasses may be cached)
            final List<OntClass> subclasses = getSubClasses(topLevelClass.getURI());
            final int numSubclasses = subclasses.size();
            if (numSubclasses == 0) {
                //logger.warn("Top level class " + topLevelClass.getLabel(null)
                //+ " has no subclases so skipping in generation");
                continue;
            }
            
            int numberResources = (int) (numSubclasses * genParams.getNumResourcesPerClass());
            numberResources = Math.min(numberResources, DataGenParams.MAX_RESOURCES);
            for (int i = 1; i <= numberResources; i++) {
                final OntClass randomSub = subclasses.get(DataGenParams.getRandom(0,
                        numSubclasses));
                if (randomSub.getURI() == null) {
                    continue;
                }
                if (randomSub.getURI().equals(OWL.Nothing.getURI())) {
                    continue;
                }
                totalCount++;
                final String resourceLabel = getResourceName(i, randomSub);
                final String resourceURI = getResourceURI(totalCount, labURI);
                generateResource(institutionURI, institutionLabel, resourceLabel, resourceURI,
                        randomSub, null, labLabel, labURI, genParams);
            }
        }
        return totalCount;
    }

    /**
     * Generates the data for a specific resource.
     * 
     * @param institutionURI
     * @param institutionLabel
     * @param resourceLabel
     * @param resourceURI
     * @param resourceClass
     * @param labName
     * @param labURI
     * @param genParams
     * @throws IOException
     */
    public abstract void generateResource(final String institutionURI,
            final String institutionLabel, final String resourceLabel, final String resourceURI,
            final OntClass resourceClass, final String resourceURL, final String labName, final String labURI,
            final DataGenParams genParams) throws IOException;

    /**
     * Creates the output directory if it does not exist.
     * 
     * @param outputPath
     * @return The created directory or null if there was an error.
     * @throws IOException
     */
    public static File ensureOutputDirectory(final String outputPath) throws IOException {
        if (outputPath == null || outputPath.length() == 0) {
            logger.error("Empty or null output path");
            return null;
        }

        final File outputDir = new File(outputPath);

        logger.info("Generating test data to directory: " + outputDir.getAbsolutePath());

        // make certain the dir exists or we can create it
        if (!outputDir.exists()) {
            if (!outputDir.mkdirs()) {
                logger.error("Failed to create output directory " + outputDir.getAbsolutePath());
                return null;
            }
        }
        return outputDir;
    }

    /**
     * Retrieves the name for a core instance given generation index.
     * 
     * @param coreNumber Number generated.
     * @param genParams Generation parameters.
     * @return Name to use.
     */
    public static String getCoreFacilityName(final String institutionName, final int coreNumber,
            final DataGenParams genParams) {
        return institutionName + " Core Facility " + coreNumber;
    }

    /**
     * Retrieves the URI for a core instance given generation index.
     * 
     * @param coreNumber Number generated.
     * @param genParams Generation parameters.
     * @return URI to use.
     */
    public static String getCoreFacilityURI(final String institutionURI, final int coreNumber,
            final DataGenParams genParams) {
        return institutionURI + "/CoreFacility/" + coreNumber;
    }

    /**
     * Retrieves the name for a research facility instance given generation
     * index.
     * 
     * @param number Number generated.
     * @param genParams Generation parameters.
     * @return Name to use.
     */
    public static String getResearchFacilityName(final String institutionName, final int number,
            final DataGenParams genParams) {
        return institutionName + " Research Facility " + number;
    }

    /**
     * Retrieves the URI for a research facility instance given generation
     * index.
     * 
     * @param number Number generated.
     * @param genParams Generation parameters.
     * @return URI to use.
     */
    public static String getResearchFacilityURI(final String institutionURI, final int number,
            final DataGenParams genParams) {
        return institutionURI + "/ResearchFacility/" + number;
    }

    /**
     * Retrieves the URI for a resource given the generation index and lab
     * resource.
     * 
     * @param resourceCount Total number of resources generated for this lab.
     * @param labURI URI of the lab
     * @return URI to use.
     */
    public static String getResourceURI(final int resourceCount, final String labURI) {
        return labURI + resourceCount;
    }

    /**
     * Retrieves the name for a resource instance given generation index and
     * class
     * 
     * @param typeCount Number generated of resources of this type generated for
     *            this lab.
     * @param resourceClass Class of the res0ource.
     * @return Name to use.
     */
    public static String getResourceName(final int typeCount, final OntClass resourceClass) {
        return resourceClass.getLabel(null) + " " + typeCount;
    }

    // cache of instrument subclasses
    private static Map<String, List<OntClass>> typeToSubclasses = new HashMap<String, List<OntClass>>();

    /**
     * Cached retrieval of subclasses for a given class.
     * 
     * @param uri
     * @return
     */
    public static List<OntClass> getSubClasses(final String uri) {
        if (typeToSubclasses.containsKey(uri)) {
            return typeToSubclasses.get(uri);
        }
        OntModel ontModel = EagleIOntDocumentManager.INSTANCE.getOntModel();
        List<OntClass> subclasses = ontModel.getOntClass(uri).listSubClasses(false).toList();
        if (subclasses == null) {
            logger.error("Subclasses are null for " + uri);
        }
        typeToSubclasses.put(uri, subclasses);
        return subclasses;
    }

    public static List<OntClass> getTypes(final OntClass ontClass, final boolean materializeHierarchy) {
        List<OntClass> types = new ArrayList<OntClass>();
        if (ontClass.getURI() == null) {
            return types;
        }
        types.add(ontClass);
        if (materializeHierarchy) {
            // get all of the parent classes
            final ExtendedIterator extIt = ontClass.listSuperClasses();
            while (extIt.hasNext()) {
                OntClass parent = (OntClass) extIt.next();
                if (parent.getURI() == null) {
                    // Some reasoners return resources with null
                    // URIs in the superclass list
                    continue;
                }
                // don't add Thing
                if (parent.getURI().equals(OWL.Thing.getURI())) {
                    continue;
                }
                if (parent.getURI().equals(RDFS.Resource.getURI())) {
                    // Some reasoners return RDFS.Resource in superclass list.
                    continue;
                }
                types.add(parent);
            }
        }
        return types;
    }
}
