package org.eaglei.search.datagen;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.jena.EagleIOntUtils;

import com.hp.hpl.jena.ontology.OntResource;

/**
 * Params for generating test data for a single institution based on the eagle-i
 * ontology.
 */
public final class DataGenParams {

    private static final int DEFAULT_MIN_CORES = 5;
    private static final int DEFAULT_MAX_CORES = 10;

    private static final int DEFAULT_MIN_RESEARCH = 5;
    private static final int DEFAULT_MAX_RESEARCH = 10;

    private static final int DEFAULT_MIN_RESOURCES = 1;
    private static final int DEFAULT_MAX_RESOURCES = 2;
    public static final int MAX_RESOURCES = 25;

    private Map<String, String> institutionURIToName = new HashMap<String, String>();
    private int coreMin = DEFAULT_MIN_CORES;
    private int coreMax = DEFAULT_MAX_CORES;
    private int researchMin = DEFAULT_MIN_RESEARCH;
    private int researchMax = DEFAULT_MAX_RESEARCH;
    private int resourcePerClassMin = DEFAULT_MIN_RESOURCES;
    private int resourcePerClassMax = DEFAULT_MAX_RESOURCES;
    private boolean materializeHierarchy = false;

    /**
     * Creates a data gen params for all institutions.
     * 
     * @return The data gen params.
     */
    public static DataGenParams createForAllInstitutions() {
        final DataGenParams params = new DataGenParams();
        // create test data for each institution
        final List<? extends OntResource> universities = EagleIOntUtils.getUniversities();
        for (Object o : universities) {
            OntResource institution = (OntResource) o;
            params.addInstitution(institution.getURI(), institution.getLabel(null));
        }
        return params;
    }

    /**
     * Creates a data gen params for a single institution.
     * 
     * @return The data gen params.
     */
    public static DataGenParams createForOneInstitution() {
        final List<? extends OntResource> universities = EagleIOntUtils.getUniversities();
        return createForOneInstitution(universities.get(0).getURI(), universities.get(0).getLabel(
                null));
    }

    /**
     * Creates a data gen params for a single specific institution.
     * 
     * @return The data gen params.
     */
    public static DataGenParams createForOneInstitution(final String uri, final String label) {
        final DataGenParams params = new DataGenParams();
        params.addInstitution(uri, label);
        return params;
    }

    private DataGenParams() {
    }

    public void addInstitution(final String institutionURI, final String institutionName) {
        this.institutionURIToName.put(institutionURI, institutionName);
    }
    
    public void clearInstitutions(){
        this.institutionURIToName.clear();
    }

    public boolean getMaterializeHierarchy() {
        return this.materializeHierarchy;
    }

    public void setMaterializeHierarchy(final boolean materializeHierarchy) {
        this.materializeHierarchy = materializeHierarchy;
    }

    public Set<String> getInstitutionURIs() {
        return this.institutionURIToName.keySet();
    }

    public String getInstitutionName(final String uri) {
        return this.institutionURIToName.get(uri);
    }

    public void setCoreFacilityRange(final int min, final int max) {
        checkRange(min, max);
        this.coreMin = min;
        this.coreMax = max;
    }

    public int getNumCoreFacilities() {
        return getRandom(this.coreMin, this.coreMax);
    }

    public void setResearchFacilityRange(final int min, final int max) {
        checkRange(min, max);
        this.researchMin = min;
        this.researchMax = max;
    }

    public int getNumResearchFacilities() {
        return getRandom(this.researchMin, this.researchMax);
    }

    public void setResourceRange(final int min, final int max) {
        checkRange(min, max);
        this.resourcePerClassMin= min;
        this.resourcePerClassMax = max;
    }

    public int getNumResourcesPerClass() {
        return getRandom(this.resourcePerClassMin, this.resourcePerClassMax);
    }

    private static void checkRange(final int lowerBound, final int upperBound) {
        assert (lowerBound >= 0);
        assert (lowerBound <= upperBound);
    }

    /**
     * Gets a random number
     */
    public static int getRandom(int lowerBound, int upperBound) {
        int delta = (int) Math.floor(Math.random() * (upperBound - lowerBound));
        return lowerBound + delta;
    }
}
