package org.eaglei.search.provider;

import java.io.IOException;
import java.util.Collection;
import java.util.HashSet;
import java.util.Set;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.model.EagleIOntConstants;
import org.eaglei.search.request.SearchRequest;
import org.eaglei.search.request.SearchResult;
import org.eaglei.search.request.SearchResultSet;

/**
 * SearchProvider utility logic.
 * 
 * @author rfrost
 */
public class SearchProviderUtil {
    
    /**
     * Retrieves the institution EIEntities from the specified SearchProvider via a SearchRequest 
     * 
     * @param provider Provider from which the institutions are retrieved.
     * 
     * @return Collection of institution EIEntities.
     * 
     * @throws IOException
     */
    public static Collection<EIEntity> getInstitutions(final SearchProvider provider) throws IOException {
        assert provider != null;
        
        // create an institution SearchRequest
        final SearchRequest institutionRequest = new SearchRequest(EIURI.create(EagleIOntConstants.UNIVERSITY_CLASS_URI));
        
        // execute the institution query against the provider
        final SearchResultSet results = provider.query(institutionRequest);
        
        // gather the unique institution entities from the results
        final Set<EIEntity> institutions = new HashSet<EIEntity>();
        for (SearchResult result: results.getResults()) {
            institutions.add(result.getEntity());
        }
        return institutions;
    }
    
    /**
     * Retrieves the institution EIEntities from the specified MultiNodeSearchProvider via a SearchRequest
     *  
     * @param provider Provider from which the institutions are retrieved.
     * 
     * @return Collection of institution EIEntities.
     * 
     * @throws IOException
     */
    public static Collection<EIEntity> getInstitutions(final MultiNodeSearchProvider provider) throws IOException {
        assert provider != null;
        
        // create an institution SearchRequest
        final SearchRequest institutionRequest = new SearchRequest(EIURI.create(EagleIOntConstants.UNIVERSITY_CLASS_URI));
        
        // execute the institution query against the provider        
        final Collection<SearchResultSet> results= provider.query(institutionRequest);
        
        // gather the unique institution entities from the results
        final Set<EIEntity> institutions = new HashSet<EIEntity>();
        for (SearchResultSet resultSet: results) {
            for (SearchResult result: resultSet.getResults()) {
                institutions.add(result.getEntity());
            }
        }
        return institutions;
    }
    
    /*
     * Gets the type binding from the SearchRequest. The term type is returned first if set, then
     * the binding type.
     */
    // TODO potentially move this logic into SearchRequest
    public static EIURI getType(final SearchRequest request) {
        EIURI type = null;
        final SearchRequest.Term term = request.getTerm();
        if (term != null && term.getURI() != null) {
            type = term.getURI();
        } else {
            final SearchRequest.TypeBinding binding = request.getBinding();
            if (binding != null) {
                type = binding.getType();
            }
        }
        return type;
    }

}
