package org.eaglei.search.request;

import java.io.Serializable;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;

/**
 * Holds the one of the results from a search. NOT thread safe.
 */
public class SearchResult implements Comparable<SearchResult>, Serializable {

    public static final long serialVersionUID = 1L;

    /*
     * The associated eagle-i resource instance.
     */
    private EIEntity entity;

    /*
     * Type of the resource. Corresponds to a class in the eagle-i ontology. 
     */
    private EIEntity type;

    /*
     * Lab at which the resource is located. This may be null if the resource is not directly associated
     * with a lab (TODO: is this valid?).
     */
    private EIEntity lab;
    
    /*
     * Institution at which the resource is located. Corresponds to an instance of the university class
     * in the eagle-i ontology.
     */
    private EIEntity institution;

    /*
     * Data type properties as defined by the eagle-i ontology.
     */
    private Map<EIURI, Set<String>> datatypeProperties;
    
    /*
     * Object properties as defined by the eagle-i ontology.
     */
    private Map<EIURI, Set<EIURI>> objectProperties;

    /*
     * Optional http URL for the search result.
     */
    private String url = null;

    /*
     * Optional match highlight
     */
    private String highlight = null;

    /*
     * Search result rank.
     */
    private float rank;
    
    private SearchResult() {}

    /**
     * Creates a new SearchResult.
     */
    public SearchResult(final EIEntity entity, final EIEntity type, final EIEntity lab, final EIEntity institution) { 
        setEntity(entity);
        setType(type);
        setLab(lab);
        setInstitution(institution);
    }

    /**
     * Retrieves the EIEntity representing the resource instance.
     * 
     * @return The EIEntity.
     */
    public EIEntity getEntity() {
        return this.entity;
    }

    /**
     * Sets the EIEntity representing the resource instance
     * @param entity The EIEntity. Cannot be null.
     */
    public void setEntity(final EIEntity entity) {
        assert entity != null;
        this.entity = entity;
    }

    /**
     * Retrieves the EIEntity representing the resource type.
     * 
     * @return The type entity. Corresponds to a class in the eagle-i ontology. 
     */
    public EIEntity getType() {
        return this.type;
    }
    
    /**
     * Sets the EIEntity representing the resource type.
     * 
     * @param type The type entity. Corresponds to a class in the eagle-i ontology. 
     */
    public void setType(final EIEntity type) {
        assert type != null;
        this.type = type;
    }

    /**
     * Retrieves the EIEntity representing the lab at which the resource is located.
     * 
     * @return The lab entity. May be null if this resource is not associated with a specific lab.
     */
    public EIEntity getLab() {
        return this.lab;
    }
    
    /**
     * Sets the lab where the resource is located.
     * 
     * @param lab EIEntity representing the resource lab. May be null.
     */
    public void setLab(final EIEntity lab) {
        this.lab = lab;
    }

    /**
     * Retrieves the EIEntity representing the resource institution.
     * 
     * @return The institution entity.
     */
    public EIEntity getInstitution() {
        return this.institution;
    }
    
    /**
     * Sets the EIEntity representing the resource institution.
     *
     * @param institution The institution entity. 
     */
    public void setInstitution(EIEntity institution) {
        this.institution = institution;
    }

    /**
     * Returns the URIs (as EIURI instances) for the result data type properties.
     * 
     * @return Set of EIURIs representing a eagle-i ontology data type properties for the
     *         result.
     */
    @SuppressWarnings("unchecked")
    public Set<EIURI> getDataTypeProperties() {
        if (this.datatypeProperties == null) {
            // GWT compiler doesn't like emptySet()
            // return Collections.emptySet();
            return Collections.EMPTY_SET;
        }
        return this.datatypeProperties.keySet();
    }

    /**
     * Adds a property binding for a data type property.
     * 
     * @param property Property URI as an EIURI. Cannot be null.
     * @param value Property value.
     */
    public void addDataTypeProperty(EIURI property, String value) {
        assert property != null;
        assert value != null;
        // Not threadsafe
        if (this.datatypeProperties == null) {
            this.datatypeProperties = new HashMap<EIURI, Set<String>>();
        }
        Set<String> values = this.datatypeProperties.get(property);
        if (values == null) {
            values = new HashSet<String>();
            this.datatypeProperties.put(property, values);
        }
        values.add(value);
    }

    /**
     * Retrieves the values of the specified data type property.
     * 
     * @param property Property URI as an EIURI.
     * @return Values of the property if it has been set or null.
     */
    public Set<String> getDataTypeProperty(EIURI property) {
        assert property != null;
        if (this.datatypeProperties == null) {
            return null;
        }
        return this.datatypeProperties.get(property);
    }
    
    /**
     * Returns the URIs (as EIURI instances) for the result object properties.
     * 
     * @return Set of EIURIs representing a eagle-i ontology properties for the
     *         result.
     */
    @SuppressWarnings("unchecked")
    public Set<EIURI> getObjectProperties() {
        if (this.objectProperties == null) {
            // GWT compiler doesn't like emptySet()
            // return Collections.emptySet();
            return Collections.EMPTY_SET;
        }
        return this.objectProperties.keySet();
    }

    /**
     * Adds a property binding for an object property.
     * 
     * @param property Property URI as an EIURI. Cannot be null.
     * @param value Property value.
     */
    public void addObjectProperty(EIURI property, EIURI value) {
        assert property != null;
        assert value != null;
        // Not threadsafe
        if (this.objectProperties == null) {
            this.objectProperties = new HashMap<EIURI, Set<EIURI>>();
        }
        Set<EIURI> values = this.objectProperties.get(property);
        if (values == null) {
            values = new HashSet<EIURI>();
            this.objectProperties.put(property, values);
        }
        values.add(value);
    }

    /**
     * Retrieves the values of the specified object property.
     * 
     * @param property Property URI as an EIURI.
     * @return Values of the property if it has been set or null.
     */
    public Set<EIURI> getObjectProperty(EIURI property) {
        assert property != null;
        if (this.objectProperties == null) {
            return null;
        }
        return this.objectProperties.get(property);
    }

    /**
     * Retrieves the HTTP URL for the result.
     * 
     * @return HTTP URL string. May be null.
     */
    public String getURL() {
        return this.url;
    }

    /**
     * Sets the HTTP URL
     * 
     * @param url HTTP URL string
     */
    public void setURL(final String url) {
        this.url = url;
    }

    /**
     * Retrieves the search highlight (for full text searches).
     * 
     * @return Highlight. May be null.
     */
    public String getHighlight() {
        return this.highlight;
    }

    /**
     * Sets the highlight.
     * 
     * @param highlight The full-text highlight.
     */
    public void setHighlight(String highlight) {
        this.highlight = highlight;
    }

    /**
     * Gets the search result rank as a float.
     * 
     * @return The result rank.
     */
    public float getRank() {
        return this.rank;
    }

    /**
     * Sets the result rank.
     * 
     * @param rank The result rank.
     */
    public void setRank(float rank) {
        this.rank = rank;
    }

    @Override
    public boolean equals(Object o) {
        if (!(o instanceof SearchResult)) {
            return false;
        }
        final SearchResult other = (SearchResult) o;
        if (!other.getEntity().equals(getEntity())) {
            return false;
        }
        return true;
    }

    @Override
    public int hashCode() {
        return getEntity().hashCode();
    }
    
    public String toString() {
        return getEntity().toString();
    }
    
    @Override
    public int compareTo(SearchResult o) {
        // Ordering based on rank, must be consistent w/ equals
        if (this.equals(o)) {
            return 0;
        } else {
            int result = (int) (getRank() - o.getRank());
            if (result != 0) {
                return result;
            } else {
                // same rank, compare based on label
                return this.getEntity().getLabel().compareTo(o.getEntity().getLabel());
            }
        }
    }
    
    //NB: For serialization tests only.  We need a way to test all the fields in one SearchResult against those from another,
    //as equals() only compares on the entity field. -Clint
    boolean deepEquals(final Object obj) {
    	//TODO: No idea if this is appropriate
    	final double delta = 0.01; 
    	
		if (this == obj) {
			return true;
		}
		if (obj == null) {
			return false;
		}
		if (getClass() != obj.getClass()) {
			return false;
		}
		SearchResult other = (SearchResult) obj;
		if (entity == null) {
			if (other.entity != null) {
				return false;
			}
		} else if (!entity.equals(other.entity)) {
			return false;
		}
		if (highlight == null) {
			if (other.highlight != null) {
				return false;
			}
		} else if (!highlight.equals(other.highlight)) {
			return false;
		}
		if (institution == null) {
			if (other.institution != null) {
				return false;
			}
		} else if (!institution.equals(other.institution)) {
			return false;
		}
		if (lab == null) {
			if (other.lab != null) {
				return false;
			}
		} else if (!lab.equals(other.lab)) {
			return false;
		}
		if (datatypeProperties == null) {
			if (other.datatypeProperties != null) {
				return false;
			}
		} else if (!datatypeProperties.equals(other.datatypeProperties)) {
			return false;
		}
		if (objectProperties == null) {
            if (other.objectProperties != null) {
                return false;
            }
        } else if (!objectProperties.equals(other.objectProperties)) {
            return false;
        }		
		//NB: Don't use Float.floatToIntBits() (as Eclipse generates) to keep GWT compiler happy
		if (Math.abs(rank - other.rank) > delta) {
			return false;
		}
		if (type == null) {
			if (other.type != null) {
				return false;
			}
		} else if (!type.equals(other.type)) {
			return false;
		}
		if (url == null) {
			if (other.url != null) {
				return false;
			}
		} else if (!url.equals(other.url)) {
			return false;
		}
		return true;
	}
}
