package org.eaglei.search.common;

import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;

/**
 * 
 * @author Clint Gilbert
 * 
 * Feb 9, 2010
 *
 * Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 *
 * @param <T>
 */
public interface Serializer<T> {
    
    String serialize(final T object) throws SerializationException;
    
    T deserialize(final String serialized) throws SerializationException;
    
    public static final Serializer<SearchRequest> SearchRequestSerializer = new Serializer<SearchRequest>() {
        @Override
        public String serialize(final SearchRequest request) throws SerializationException {
            assert request != null;
            
            //TODO: How are errors handled? returning null?
            return request.toURLParams();
        }
        
        @Override
        public SearchRequest deserialize(final String serialized) throws SerializationException {
            assert serialized != null;
            
            final SearchRequest request = new SearchRequest(serialized);
            
            //TODO: How are errors handled? Is null an error?
            if(request == null) {
                throw new SerializationException("Error unmarshalling SearchRequest; got null result for: '" + serialized + "'");
            }
            
            return request;
        }
    };
    
    public static final Serializer<SearchResultSet> SearchResultSetSerializer = new Serializer<SearchResultSet>() {
        @Override
        public String serialize(final SearchResultSet resultSet) throws SerializationException {
            return GsonHolder.Gson.toJson(resultSet, SearchResultSet.class);
        }
        
        @Override
        public SearchResultSet deserialize(final String serialized) throws SerializationException {
            return GsonHolder.Gson.fromJson(serialized, SearchResultSet.class);
        }
    };
    
    public static final Serializer<SearchCountRequest> SearchCountRequestSerializer = new Serializer<SearchCountRequest>() {
        @Override
        public String serialize(final SearchCountRequest request) throws SerializationException {
            return GsonHolder.Gson.toJson(request, SearchCountRequest.class);
        }
        
        @Override
        public SearchCountRequest deserialize(final String serialized) throws SerializationException {
            return GsonHolder.Gson.fromJson(serialized, SearchCountRequest.class);
        }
    };
    
    public static final Serializer<SearchCounts> SearchCountsSerializer = new Serializer<SearchCounts>() {
        @Override
        public String serialize(final SearchCounts counts) throws SerializationException {
            return GsonHolder.Gson.toJson(counts, SearchCounts.class);
        }
        
        @Override
        public SearchCounts deserialize(final String serialized) throws SerializationException {
            return GsonHolder.Gson.fromJson(serialized, SearchCounts.class);
        }
    };    
    
    public static final Serializer<String> NullSerializer = new Serializer<String>() {
        @Override
        public String deserialize(final String serialized) throws SerializationException {
            return serialized;
        }

        @Override
        public String serialize(final String object) throws SerializationException {
            return object;
        }
    };
}
