package org.eaglei.search.provider.lucene;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.config.SearchProperties;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;

/**
 * Implementation of SearchProvider that dynamically builds a Lucene index using the results
 * retrieved from another SearchProvider.
 * 
 * @author frost
 */
public class CompositeLuceneProvider implements SearchProvider {
    
    private static final Log logger = LogFactory.getLog(CompositeLuceneProvider.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    private final EIOntModel eagleiOntModel;
    /*
     * SearchProvider that is queried to populated an embedded Lucene index 
     */
    private final SearchProvider nestedProvider;
    /*
     * SearchProvider for the embedded Lucene index
     */
    private LuceneSearchProvider luceneWrapper;

    /**
     * Creates a new CompositeLuceneProvider that wraps the specified SearchProvider
     * @param provider Nested SearchProvider
     * @throws IOException
     */
    public CompositeLuceneProvider(final EIOntModel eagleiOntModel, final SearchProvider provider) throws IOException {
        this.eagleiOntModel = eagleiOntModel;
        this.nestedProvider = provider;
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#init()
     */
    public void init() throws IOException {

        // initialize the nested SearchProvider
        
        if (DEBUG) {
            logger.debug("Initializing nested provider for CompositeLuceneProvider");
        }
        this.nestedProvider.init();
        if (DEBUG) {
            logger.debug("Nested provider initialized");
        }
        
        // create the Lucene index (Directory & Analyzer), indexer, provider and updater
        
        final Directory dir = LuceneSearchProviderFactory.getDirectory();
        final Analyzer analyzer = LuceneSearchProviderFactory.getAnalyzer();
        final LuceneSearchProviderIndexer indexer = new LuceneSearchProviderIndexer(eagleiOntModel, analyzer, dir); 
        indexer.commit();
        this.luceneWrapper = new LuceneSearchProvider(eagleiOntModel, dir, analyzer);
        final LuceneSearchProviderIndexUpdater indexerUpdater = new LuceneSearchProviderIndexUpdater(this.nestedProvider, indexer);
        
        // check if the System property controlling use of an indexing thread was set
        // and either spawn a thread to handle index updates or directly update the index once (this is 
        // typically used just for unit testing).
        
        boolean useIndexerThread = Boolean.valueOf(System.getProperty(SearchProperties.USE_INDEXER_THREAD, Boolean.TRUE.toString()));
        if (useIndexerThread) {
            Thread initThread = new Thread(indexerUpdater, "CompositeLuceneProviderIndexUpater");
            initThread.setPriority(Thread.MIN_PRIORITY);
            initThread.setDaemon(true);
            initThread.start();
        } else {
            indexerUpdater.updateIndex();
        }
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {
        return this.luceneWrapper.query(request);
    }
    
    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#count(org.eaglei.search.provider.SearchCountRequest)
     */
    public SearchCounts count(SearchCountRequest request) throws IOException {
        return this.luceneWrapper.count(request);
    }
}
