package org.eaglei.search.provider.lucene;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TopFieldDocs;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.store.Directory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;

import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 * SearchProvider that queries a Lucene index populated according to the 
 * schema defined in LuceneSearchIndexSchema.
 * @author frost
 */
public final class LuceneSearchProvider extends LuceneSearchIndexSchema implements SearchProvider { 

    private static final Log logger = LogFactory.getLog(LuceneSearchProvider.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    /*
     * Controls whether datatype and object properties are returned.
     */
    private static final boolean RETRIEVE_PROPERTIES = false;

    /*
     * Handle to the in-memory eagle-i ontology
     */
    private final EIOntModel eagleiOntModel;
    /*
     * The Lucene index. This may be in-memory or file-based.
     */
    private Directory dir;
    /*
     * The Lucene analyzer used for both indexing and query execution.
     */
    private Analyzer analyzer;
    /**
     * Helper class that creates a Query from a SearchRequest
     */
    private LuceneQueryBuilder queryBuilder;
    /**
     * Cache of the URIs for preferred label properties
     */
    private List<EIURI> prefLabelProperties = new ArrayList<EIURI>();

    /**
     * Creates a LuceneProvider that executes SearchRequests over the specified Directory using the
     * specified Analyzer. The Directory must be populated using LuceneIndexer.
     * @param eagleiOntModel Reference to the eagle-i ontology model.
     * @param dir Directory holding the Lucene index.
     * @param analyzer Analyzer to use for query execution.
     */
    public LuceneSearchProvider(final EIOntModel eagleiOntModel, final Directory dir, final Analyzer analyzer) throws IOException {
        assert dir != null;
        assert analyzer != null;        
        this.eagleiOntModel = eagleiOntModel;
        this.dir = dir;
        this.analyzer = analyzer;
        this.queryBuilder = new LuceneQueryBuilder(eagleiOntModel, analyzer);
        retrieveOntologyMetadata();
    }
    
    @Override
    public void init() throws IOException {
        // no-op
    }
    
    /*
     * Retrieves various metadata from the eagle-i ontology that is cached as
     * instance vars in this provider and reused on queries.
     */
    private void retrieveOntologyMetadata() {
        // properties used to compute preferred labels
        List<Property> props = ((JenaEIOntModel) eagleiOntModel).getPrefLabelProperties();
        for (Property prop: props) {
            this.prefLabelProperties.add(EIURI.create(prop.getURI()));
        }
    }

    /* (non-Javadoc)
     * @see org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request.SearchRequest)
     */
    public SearchResultSet query(final SearchRequest request) throws IOException {
        return query(request, true);
    }
        
    /*
     * Version of query that allows the optional creation of search results. This is supported for the
     * count use case (when retrieving counts, creation of actual results is not necessary).
     * @param request The search request
     * @param createResults True to create SearchResult objects. False to just execute the query and get counts.
     */
    private SearchResultSet query(final SearchRequest request, final boolean createResults) throws IOException {
        assert request != null;

        // create the query
        Query query = null;
        try {
            query = this.queryBuilder.createQuery(request);
        } catch (ParseException pe) {
            throw new IOException(pe.getLocalizedMessage());
        }

        // if there was an error creating the query, just return an empty result set
        if (query == null) {
            return new SearchResultSet(request);
        }
        
        // execute the search
        return executeSearch(request, query, createResults);
    }
    
    @Override
    public SearchCounts count(SearchCountRequest request) throws IOException {
        assert request != null;
        final SearchRequest searchRequest = request.getRequest();
        final SearchCounts counts = new SearchCounts(searchRequest);

        // execute a count-only query for each count type
        for (EIURI type: request.getCountTypes()) {
            SearchRequest countRequest = new SearchRequest(searchRequest.toURLParams());
            if (type == null) {
                countRequest.setBinding(null);
            } else {
                countRequest.setBinding(new SearchRequest.TypeBinding(type));
            }
            // pass in false to so we don't create results
            final SearchResultSet results = query(countRequest, false); 
            counts.setClassCount(type, results.getTotalCount());
        }
        return counts;
    }    
    
    /*
     * Executes the Lucene query and creates SearchResults from the output docs.
     * @param request The SearchRequest
     * @param query The Lucene Query
     * @param createResults True if results should be created. False to just get a count.
     */
    private SearchResultSet executeSearch(final SearchRequest request, final Query query, final boolean createResults)
    throws IOException {
        
        final SearchResultSet results = new SearchResultSet(request);
        
        // create an IndexSearcher
        // TODO would like to create and reuse a single IndexSearcher but it is not seeing index changes on
        // reopen...
        final IndexSearcher searcher = new IndexSearcher(this.dir, true);
        searcher.setDefaultFieldSortScoring(true, true);
        
        // Execute the Query using the IndexSearcher
        final TopFieldDocs docs = searcher.search(query, null, request.getStartIndex() + request.getMaxResults(), Sort.RELEVANCE);
        
        if (DEBUG) {
            logger.debug("Found " + docs.totalHits + " matches");
        }
        
        // Set the count data in the SearchResultSet
        results.setTotalCount(docs.totalHits);
        results.setStartIndex(request.getStartIndex());
        
        // if not populating individual SearchResults, return with the counts        
        if (!createResults) {
            return results;
        }

        // create a highligher for the query
        final Highlighter highlighter = new Highlighter(new QueryScorer(query)); 
        
        // get page subset and create a SearchResult for each Document in the page
        final int cap = request.getStartIndex()+ request.getMaxResults();
        for (int i = request.getStartIndex(); (i < cap) && (i < docs.scoreDocs.length); i++) {

            // get the Lucene ScoreDoc and Document
            final ScoreDoc scoreDoc = docs.scoreDocs[i];
            final Document document = searcher.doc(scoreDoc.doc);
            
            // create a SearchResult from the document fields
            final SearchResult result = createSearchResultForDocument(searcher, scoreDoc, document);
            
            // if there was an issue creating the result, skip
            if (result == null) {
                continue;
            }

            // compute the highlight
            final String highlight = LuceneHighlightGenerator.computeHighlight(highlighter, analyzer, eagleiOntModel, request, query, document, result);
            if (highlight != null) {
                result.setHighlight(highlight);
            }

            // check for dups in the resultset
            if (results.getResults().contains(result)) {
                logger.error("Found duplicate result");
            } 
            
            // add the result
            results.getResults().add(result);
        }
    
        return results;
    }
    
    /**
     * Creates a SearchResult for a single Lucene Document match
     * @param searcher IndexSearcher used to execute the query.
     * @param scoreDoc The ScoreDoc for the match
     * @param document The Document for the match
     * @return The SearchResult
     */
    private SearchResult createSearchResultForDocument(final IndexSearcher searcher, final ScoreDoc scoreDoc, final Document document) {

        // get the document score
        final float score = scoreDoc.score;
        
        // get the resource URI
        final String resource = document.get(LuceneSearchProviderIndexer.URI);

        // get the resource label (via the priority ordering of label properties)
        String label = null;
        for (EIURI prop: this.prefLabelProperties) {
            String[] values = document.getValues(prop.toString());
            if (values.length > 0) {
                label = values[0];
                break;
            }
        }
        if (label == null) {
            return null;
        }
        
        // create an EIEntity for the resource
        final EIEntity resourceEntity = EIEntity.create(EIURI.create(resource), label);

        // get the EIInstitution 
        final String institution_uri = document.get(LuceneSearchProviderIndexer.INSTITUTION_URI);
        final String institution_label = document.get(LuceneSearchProviderIndexer.INSTITUTION_LABEL);
        final EIEntity institutionEntity = EIEntity.create(EIURI.create(institution_uri),
                institution_label);

        // get the EIClass
        // TODO this is just getting the first type...
        final String type = document.get(RDF.type.getURI() + LuceneSearchProviderIndexer.OBJECT_URI_POSTFIX);
        if (type == null) {
            logger.error("Null rdf:type for " + resource);
            return null;
        }
        final EIClass eiClass = eagleiOntModel.getClass(EIURI.create(type));
        if (eiClass == null) {
            logger.error("Unable to locate resource class " + type + " for " + resource);
            return null;
        }

        // create the SearchResult
        final SearchResult result = new SearchResult(resourceEntity, eiClass.getEntity(), null, institutionEntity);

        // add all datatype and object properties
        for (Fieldable f: document.getFields()) {
            addResultProperty(result, document, f);
        }

        result.setURL(resource);
        result.setRank(score);

        return result;
    }
    
    /**
     * Potentially adds a match Document field to the SearchResult if it represents an RDF datatype or object property.  
     * @param result The SearchResult 
     * @param field The Document field
     */
    private static void addResultProperty(final SearchResult result, final Document document, final Fieldable f) {
        final String name = f.name();
        final String strValue = f.stringValue();
        // check name against the known fields
        if (strValue != null && isPropertyField(name)) {
            // if tokenized, it is a datatype property
            if (f.isTokenized()) {
                //logger.debug("Adding data type property " + name + " and value " + strValue);
                if (RETRIEVE_PROPERTIES) {
                    result.addDataTypeProperty(EIURI.create(name), strValue);
                }
                if (LuceneSearchProviderIndexer.INDEX_OBJECT_PROP_LABELS && isLabProperty(name)) {
                    String labURI = document.get(name + LuceneSearchProviderIndexer.OBJECT_URI_POSTFIX);
                    if (labURI != null) {
                        final EIEntity labEntity = EIEntity.create(EIURI.create(labURI), strValue);
                        //logger.debug("Found lab: " + labEntity);
                        result.setLab(labEntity);
                    }
                }
            } else {
                //logger.debug("Adding object property " + name + " and value " + strValue);
                EIURI propURI = EIURI.create(LuceneSearchProviderIndexer.stripObjectURIPostfix(name));
                if (RETRIEVE_PROPERTIES) {
                    result.addObjectProperty(propURI, EIURI.create(strValue));
                }
                if (!LuceneSearchProviderIndexer.INDEX_OBJECT_PROP_LABELS) {
                    logger.warn("Must reimplement dynamic lab computation if not indexing object property labels");
                    /*
                    if (isLabProperty(propURI.toString())) {
                        addLab(result, strValue);
                    }
                    */
                }
            }
        }        
    }
    
    /*
     * Adds a lab EIEntity to the SearchResult for the specified URI. Retrieves the label via a search
     * against the index.
     */
    /*
    private void addLab(final SearchResult result, final String labURI) throws IOException {
        final BooleanQuery query = new BooleanQuery();
        final PhraseQuery resourceQuery = new PhraseQuery();
        resourceQuery.add(new Term(LuceneSearchProviderIndexer.URI, labURI));
        query.add(resourceQuery, BooleanClause.Occur.MUST);
        final IndexSearcher searcher = new IndexSearcher(this.dir, true);        
        searcher.setDefaultFieldSortScoring(true, true);
        final TopFieldDocs docs = searcher.search(query, null, 1, Sort.RELEVANCE);
        //logger.debug("Found " + docs.totalHits + " lab matches");
        if (docs.totalHits > 0) {
            final ScoreDoc scoreDoc = docs.scoreDocs[0];
            final Document document = searcher.doc(scoreDoc.doc);
            final float score = scoreDoc.score;
            final String resource = document.get(LuceneSearchProviderIndexer.URI);
            String label = null;
            for (EIURI prop: this.prefLabelProperties) {
                String[] values = document.getValues(prop.toString());
                if (values.length > 0) {
                    label = values[0];
                    break;
                }
            }
            final EIEntity labEntity = EIEntity.create(EIURI.create(resource), label);
            //logger.debug("Found lab: " + label);
            result.setLab(labEntity);
        }
    }
    */

}
