package org.eaglei.search.provider.lucene;

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.eaglei.lexical.lucene.PorterStemAnalyzer;
import org.eaglei.search.config.SearchProperties;

/**
 * Extension of SearchProperties that contains properties for the LuceneProvider.
 * @author frost 
 */
public final class LuceneSearchProviderProperties extends SearchProperties {
    
    private static final Log logger = LogFactory.getLog(LuceneSearchProviderProperties.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    /** 
     * System property that holds the index update frequency in msec.
     */
    public static final String UPDATE_FREQUENCY = "org.eaglei.search.provider.lucene.composite.update.freq";  
    /**
     * Default update frequency 
     */
    public static final long DEFAULT_UPDATE_FREQ = 1000*10; // every 10 seconds
    /**
     * Name of SearchConfiguration property that holds a directory on classpath where provider looks for lucene index
     */    
    public static final String LUCENE_INDEX_DIR_PROP = "org.eaglei.search.provider.lucene.index.dir";

    
    /**
     * Retrieves the Lucene Directory to use as a search index.
     * 
     * @return Lucene Directory.
     * 
     * @throws IOException Thrown if an error is encountered.
     */
    public static Directory getDirectory() throws IOException {
        final String indexDir = System.getProperty(LUCENE_INDEX_DIR_PROP);
        
        // if the index directory property is not null, create an FSDirectory for that
        if (indexDir != null) {
            final File dir = new File(indexDir);
            if (DEBUG) {
                logger.debug("Using Lucene index directory: " + dir.getAbsolutePath());
            }
            if (!dir.exists()) {
                dir.mkdirs();
            } else if (!dir.isDirectory()) {
                logger.error("Specified Lucene index directory is not a directory");
                return new RAMDirectory();       
            }
            return FSDirectory.open(dir);
        } 
        if (DEBUG) {
            logger.debug("Using Lucene RAMDirectory");
        }
        return new RAMDirectory();
    }
    
    /**
     * Creates the Lucene Analyzer to use for searching and indexing.
     * @return Analyzer.
     */
    public static Analyzer getAnalyzer() {
        return new PorterStemAnalyzer();
    }
}
