package org.eaglei.search.provider.lucene;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.InvalidTokenOffsetsException;

import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;

import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;

import com.hp.hpl.jena.vocabulary.RDF;

/**
 * Utility class that holds logic for computing highlights for SearchRequest matches against the Lucene index
 * created per LuceneSearchIndexSchema. 
 */
public final class LuceneHighlightGenerator { 

    private static final Log logger = LogFactory.getLog(LuceneHighlightGenerator.class);
    
    /*
     * Maximum number of highlight field matches to return
     */
    private static final int MAX_HIGHLIGHT_MATCHES = 3;
    
    /**
     * Computes the highlight for a search request.
     * @param highlighter The highlighter to use for computation
     * @param analyzer The Lucene analyzer used to generate the match
     * @param ontModel Reference to the eagle-i ontology model.
     * @param request The search request
     * @param query The Lucene query
     * @param document Matching document
     * @param result The SearchResult
     * @return Computed highlight
     */
    protected static String computeHighlight(final Highlighter highlighter, final Analyzer analyzer, final EIOntModel ontModel, final SearchRequest request, final Query query, final Document document, final SearchResult result) {
        // compute a highlight if this was a term query or had a datatype binding
        SearchRequest.Term term = request.getTerm();
        SearchRequest.TypeBinding binding = request.getBinding();
        if ((term != null && term.getQuery() != null) || (binding != null && !binding.getDataTypeProperties().isEmpty())) {
            // if we can locate the associated property & label, prepend that
            // TODO, try and re-enable, encountering performance problems
            final StringBuilder highlights = new StringBuilder();
            boolean firstHighlight = true;
            int totalHighlightMatches = 0;
            for (Fieldable f: document.getFields()) {
                final String name = f.name();
                final String strValue = f.stringValue();
                if (strValue != null && LuceneSearchIndexSchema.isPropertyField(name) && f.isTokenized()) {
                    String highlight = getHighlightForField(highlighter, analyzer, name, strValue);
                    if (highlight != null) {
                        if (firstHighlight) {
                            highlights.append("...");
                            firstHighlight = false;
                        }
                        // optionally prepend the property label
                        final String propLabel = getPropertyLabel(ontModel, result.getType().getURI().toString(), name);
                        if (propLabel != null) {
                            highlights.append(propLabel + ": ");
                        }  else {
                            //logger.error("Could not find label for " + name);
                        }
                        highlights.append(highlight);
                        highlights.append("...");
                        
                        // check if we've the max allowed field matches
                        if (totalHighlightMatches++ >= MAX_HIGHLIGHT_MATCHES) {
                            break;
                        }
                    }
                }
            }
            return highlights.toString();
        }
        
        return null;
    }
    
    /*
     * Computes the label for the specified property of the specified class
     * @param typeURI URI of ontology class
     * @param propURI URI of property
     * @return The property label.
     */
    private static String getPropertyLabel(final EIOntModel eagleiOntModel, final String typeURI, final String propURI) {
        // TODO fix this in JenaEIOntModel
        // special handling for rdf:type
        if (propURI.equals(RDF.type.getURI())) {
            return "type";
        }
        
        for (EIProperty prop: eagleiOntModel.getProperties(EIURI.create(typeURI))) {
            if (prop.getEntity().getURI().toString().equals(propURI)) {
                return prop.getEntity().getLabel();
            }
        }
        return null;
    }
    

    
    /*
     * Computes the match highlight for the specified field and field value.
     * @param highlighter The Lucene highlighter to use for highlight computation
     * @param fieldName Name of the index field
     * @param field value Value of the field
     * @return Computed highlight.
     */
    private static String getHighlightForField(final Highlighter highlighter, final Analyzer analyzer, final String fieldName, final String fieldValue) {
        try {
            final String highlight = highlighter.getBestFragment(analyzer, fieldName, fieldValue); //MAX_FRAGMENTS);
            if (highlight != null) {
                return highlight;
            }
        } catch (InvalidTokenOffsetsException itoe) {
            logger.error(itoe);
        } catch (IOException ioe) {
            logger.error(ioe);
        }
        return null;
    }
}
