package org.eaglei.search.provider.lucene;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexReader;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.Collector;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Scorer;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.datagen.AbstractGenerator;
import org.eaglei.search.events.ChangeEventPayloadImpl;
import org.eaglei.search.events.ChangeEventPayload;
import org.eaglei.search.events.ChangeEventPayloadItem;
import org.eaglei.search.events.IndexChangeProcessor;
import org.eaglei.search.events.IndexChangeEvent;

import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 * Creates a Lucene index for eagle-i RDF resource data according to the schema defined in LuceneSearchIndexSchema.
 * @author frost
 */
public final class LuceneSearchProviderIndexer extends LuceneSearchIndexSchema 
					implements IndexChangeProcessor, Runnable {

    private static final Log logger = LogFactory.getLog(LuceneSearchProviderIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();

    /*
     * Maximum number of results to retrieve underlying institutional node(s).
     * TODO support retrieval of ALL results
     */
    private static final int MAX_RESULTS_TO_INDEX_FROM_EACH_NODE = 100000;
    
    /**
     * Cache of the URIs for preferred label properties
     */
    private List<EIURI> prefLabelProperties = new ArrayList<EIURI>();
    /**
     * Reference to the in-memory model of the eagle-i ontology
     */
    private final EIOntModel eagleiOntModel;
    /**
     * Handle to the index writer
     */
    private final IndexWriter iwriter;
    
    /**
     * Notation for an embedded class.
     */
    private static final String EMBEDDED_CLASS = "http://eagle-i.org/ont/app/1.0/ClassGroup_embedded_class";

    /*
     * List of index changes to be processed.
     */
//    private List<IChangeEventPayload> eventPayloads = new ArrayList<IChangeEventPayload>();
    
    private final SearchProvider nestedProvider;
    
    private long updateFrequency = LuceneSearchProviderProperties.DEFAULT_UPDATE_FREQ; 
    
    /**
     * Creates the LuceneSearchProviderIndexer
     * 
     * @param eagleiOntModel Referenced to the eagle-i ontology
     * @param analyzer The Lucene analyzer that is used for indexing and searching.
     * @param directory The directory that holds the index.
     * 
     * @throws IOException Thrown if an error is encountered.
     */
    public LuceneSearchProviderIndexer(final EIOntModel eagleiOntModel, final Analyzer analyzer, 
    						final Directory directory, final SearchProvider nestedProvider) throws IOException {
        this.eagleiOntModel = eagleiOntModel;
        this.iwriter = new IndexWriter(directory, analyzer, IndexWriter.MaxFieldLength.LIMITED);
        retrieveOntologyMetadata();        
        
        this.nestedProvider = nestedProvider;
        final String updateFrequencyProp = System.getProperty(LuceneSearchProviderProperties.UPDATE_FREQUENCY); 

      if (updateFrequencyProp != null) {
          try {
              setUpdateFrequency(Long.parseLong(updateFrequencyProp));
          } catch (NumberFormatException nfe) {
              // default
          }
      } // default
    }

    /**
     * Sets the frequency for updating the lucene index from the embedded provider.
     * @param updateFrequency Update frequency in msec. 
     */
    public void setUpdateFrequency(final long updateFrequency) {
        // must be longer than 1 second 
        if (updateFrequency < 1000) {
            this.updateFrequency = 1000;
        } else {
        	this.updateFrequency = updateFrequency;
        }
    }
    
    /**
     * @see #setUpdateFrequency(long)
     */
    public long getUpdateFrequency() {
        return this.updateFrequency;
    }
    
    /*
     * Retrieves various metadata from the eagle-i ontology that is cached as
     * instance vars in this provider and reused on queries.
     */
    private void retrieveOntologyMetadata() {
        // properties used to compute preferred labels
        List<Property> props = ((JenaEIOntModel) eagleiOntModel).getPrefLabelProperties();
        for (Property prop: props) {
            this.prefLabelProperties.add(EIURI.create(prop.getURI()));
        }
    }
    
    /**
     * Retrieves the IndexWriter
     * @return
     */
    public IndexWriter getIndexWriter() {
        return this.iwriter;
    }

    /**
     * Commits any pending changes the changes
     * @throws IOException
     */
    public void commit() throws IOException {
        iwriter.optimize();
        iwriter.commit();
    }
    
    /**
     * Gets the EIURIs of all documents that reference the specified document via an 
     * object property.
     * @param uri URI of property whose referencing documents are being retrieved.
     * @return List of URIs of referencing documents.
     * @throws IOException Thrown if an error is encountered executing the query
     */
    public List<EIURI> getRelatedDocuments(final EIURI uri) throws IOException {

        // create the Lucene query 
        final BooleanQuery query = new BooleanQuery();
        final PhraseQuery propQuery = new PhraseQuery();
        propQuery.add(new Term(RELATED, uri.toString()));
        query.add(propQuery, BooleanClause.Occur.MUST);
        
        // create an IndexSearcher
        final IndexSearcher searcher = new IndexSearcher(this.iwriter.getDirectory(), true);
        searcher.setDefaultFieldSortScoring(true, true);

        // collector that grabs the URIs of all retrieved Documents
        final List<EIURI> uris = new ArrayList<EIURI>();
        final Collector collector = new Collector(){
            IndexReader reader;
            int docbase;
            public void setNextReader(IndexReader reader, int docbase) throws IOException {
                this.reader = reader;
                this.docbase = docbase;
            }
            public void collect(final int doc) throws IOException {
                Document document = this.reader.document(this.docbase + doc);
                uris.add(EIURI.create(document.get(URI)));
            }
            
            public boolean acceptsDocsOutOfOrder() {
                return true;
            }
            public void setScorer(Scorer scorer) throws IOException {
                // no-op
            }
        };
        // execute the search
        searcher.search(query, null, collector);
        return uris;        
    }
    
    /**
     * Removes the _uri postfix from the document field name.
     * 
     * @param fieldWithPostfix Field with the _uri postfix
     * @return Field name without the _uri postfix
     */
    protected static String stripObjectURIPostfix(final String fieldWithPostfix) {
        assert fieldWithPostfix != null;
        if (!fieldWithPostfix.endsWith(OBJECT_URI_POSTFIX)) {
            return fieldWithPostfix;
        }
        return fieldWithPostfix.substring(0, fieldWithPostfix.length() - OBJECT_URI_POSTFIX.length());
    }
    
    /**
     * Updates the document with the specified URI to add object property labels.
     * 
     * @param uri URI of document to update with fields for the labels of resources connected via object properties.
     * @throws IOException Thrown if an error is encountered.
     */
    private void addIndirectProperties(final EIURI uri) throws IOException {
        //logger.debug("Adding indirect properties for " + uri);
        final Document document = getDocumentByURI(uri);
        if (document == null) {
            logger.debug("Failed to find " + uri + " in index");
            return;
        }
        
        // remove the text field, this stores the indirect properties; it also stores all the direct
        // datatype properties, the institution name and the names of the types so need to add those back
        document.removeFields(TEXT);
        
        // retrieve all of the properties that hold object prop URIs to value URIs
        final Map<String, String> objectPropURIToValue = new HashMap<String, String>();
        final Map<String, String> datatypePropURIToValue = new HashMap<String, String>();
        for (Fieldable f: document.getFields()) {
            final String name = f.name();
            final String strValue = f.stringValue();
            // check name against the known fields
            if (strValue != null && LuceneSearchProvider.isPropertyField(name)) {
                if (!f.isTokenized()) {
                    // remove the postfix
                    final String objectPropURI = stripObjectURIPostfix(name);
                    objectPropURIToValue.put(objectPropURI, strValue);
                } else {
                    // this is a datatype property, need to save so that it can
                    // be added back to the text field
                    datatypePropURIToValue.put(name, strValue);
                }
            }
        }
        
        // add all of the datatype props back to the text field
        for (String value: datatypePropURIToValue.values()) {
            addToText(document, value);
        }

        // remove all properties that hold object prop resource text
        for (String propURI: objectPropURIToValue.keySet()) {
            document.removeFields(propURI);
        }
        
        // update the object prop resource text
        for (String propURI: objectPropURIToValue.keySet()) {
            final String propValue = objectPropURIToValue.get(propURI);
            // find the document for the object prop resource in the
            // index
            Document objectDoc = getDocumentByURI(EIURI.create(propValue));
            if (objectDoc != null) {
                for (Fieldable prefTextField: objectDoc.getFieldables(PREF_TEXT)) {
                    final String prefText = prefTextField.stringValue();
                    //logger.debug("Adding text " + prefText + " for object prop " + propURI);
                    
                    // add the pref text fields for this resource to the text field
                    addToText(document, prefText);
                    
                    // add the pref text fields for this resource to a prop-specific field
                    Field objectPropLabel= new Field(propURI.toString(), prefText, Field.Store.YES, Field.Index.ANALYZED);
                    objectPropLabel.setBoost(LOW_BOOST);
                    document.add(objectPropLabel);                            
                }
            } else {
                // look up as an ontology class
               final EIClass eiClass = eagleiOntModel.getClass(EIURI.create(propValue));
               if (eiClass != null) {
                   final String prefText = eiClass.getEntity().getLabel();
                   // add the pref text fields for this resource to the text field
                   addToText(document, prefText);
                   // add the pref text fields for this resource to a prop-specific field
                   Field objectPropLabel= new Field(propURI.toString(), prefText, Field.Store.YES, Field.Index.ANALYZED);
                   objectPropLabel.setBoost(LOW_BOOST);
                   document.add(objectPropLabel);                            
               } else {
                   //logger.error("Did not find document or class for " + propValue);
               }
            }
        }
        
        // add the institution label back to text
        addToText(document, document.get(INSTITUTION_LABEL));

        // add the types back to the text
        String typeURI = document.get(RDF.type.getURI() + OBJECT_URI_POSTFIX);
        indexTypes(document, EIURI.create(typeURI), true, false);
        
        // update the document
        iwriter.updateDocument(new Term(URI, uri.toString()), document);
    }
    
    /*
     * Retrieves the relevant document by URI
     * @param uri URI of resource to retrieve.
     * @return Associated Lucene Document.
     */
    public Document getDocumentByURI(final EIURI uri) throws IOException {
        // create a query 
        final PhraseQuery propQuery = new PhraseQuery();
        propQuery.add(new Term(URI, uri.toString()));
        
        final IndexSearcher searcher = new IndexSearcher(this.iwriter.getDirectory(), true);
        searcher.setDefaultFieldSortScoring(true, true);
        
        final TopDocs docs = searcher.search(propQuery, 1);
        if (docs.totalHits == 0) {
            //logger.error("Did not find " + uri + " in search index");
            return null;
        }
        final ScoreDoc scoreDoc = docs.scoreDocs[0];
        final Document document = searcher.doc(scoreDoc.doc);
        return document;
    }
    

    private void updateDocument(EIURI eiURI, Document doc) throws IOException {
    	try {
			iwriter.updateDocument(new Term(URI, eiURI.toString()), doc);
		} catch (CorruptIndexException e) {
			// TODO Auto-generated catch block
			e.printStackTrace();
		}
    }
    

    /*
     * If there is a document with the specified URI, remove from index
     * @param uri URI of resource to remove from the index.
     */
    private void deleteDocumentByURI(final EIURI uri) throws IOException {
        final PhraseQuery query = new PhraseQuery();
        query.add(new Term(URI, uri.toString()));
        this.iwriter.deleteDocuments(query);        
    }
    
    /**
     * Checks if this SearchResult represents a deleted resource. The /harvest API returns a special
     * resource representation for resources that have been deleted since the specified timestamp. 
     * @return True if it represents a deleted resource.
     */
    protected static boolean isDeletedSearchResult(final ChangeEventPayloadItem result) {
        if (result.getType().getURI().toString().equals(EIOntConstants.IS_DELETED)) {
            return true;
        }
        return false;
    }
    

    
    public void processIndexChangeEvent(IndexChangeEvent e) throws IOException {
        if (e == null) {
            return;
        }
        
        ChangeEventPayload payload = e.getPayload();
        processPayload(payload);
    }
    
    /**
     * Indexes the specified SearchResult.
     * @param result SearchResult
     * @param materializeTypes True if the types should be materialized.
     * @throws IOException Thrown if an error is encountered indexing the result
     */
    public void indexSearchResult(final ChangeEventPayloadItem result, final boolean materializeTypes) throws IOException {
        
        final EIURI uri = result.getEntity().getURI(); 
        
        deleteDocumentByURI(uri);
        
        // if the type of the result is "isDeleted", don't add again
        if (isDeletedSearchResult(result)) {
            return;
        }
        
        // create a Lucene document for the resource
        final Document doc = new Document();

        // index the URI
        doc.add(new Field(URI, uri.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        
        // add the institution URI and label
        final EIEntity institutionEntity = result.getInstitution();
        doc.add(new Field(INSTITUTION_URI, institutionEntity.getURI().toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        doc.add(new Field(INSTITUTION_LABEL, institutionEntity.getLabel(), Field.Store.YES, Field.Index.ANALYZED));        
        
        // index the institution label
        addToText(doc, institutionEntity.getLabel());

        final EIEntity typeEntity = result.getType();
        
        // is this an eagle-i resource?
        final EIClass typeClass = eagleiOntModel.getClass(typeEntity.getURI());
        if (typeClass == null) { 
            logger.error("Resource " + result.getEntity() + " with type " + typeClass + " is not a valid eagle-i class");
            return;
        }
        doc.add(new Field(RESOURCE_FLAG, String.valueOf(typeClass.isEagleIResource()), Field.Store.YES, Field.Index.NOT_ANALYZED));   
        
        // index the asserted type
        // TODO handle multiple asserted types 
        doc.add(new Field(RDF.type.getURI() + OBJECT_URI_POSTFIX, typeEntity.getURI().toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));

        // add the types (potentially with materialization)
        // TODO retrieve the inferred types from the repository so we don't need to materialize here
        indexTypes(doc, typeEntity.getURI(), materializeTypes, true);

        indexProperties(result, doc);

        // add the document to the to the index
        this.iwriter.addDocument(doc);
    }
    
    /**
     * Index the data type and object properties for a given search result and document.
     * 
     * @param result SearchResult
     * @param doc Document
     */
    public void indexProperties(ChangeEventPayloadItem result, Document doc) {
        // index each of the data type properties
        for (EIURI propURI: result.getDataTypeProperties()) {
            // add preferred label properties to the pref_text field
            boolean addToPrefText = prefLabelProperties.contains(propURI); 
            
            final Set<String> values = result.getDataTypeProperty(propURI);
            for (String value: values) {
                // index the property value using the URI
                doc.add(new Field(propURI.toString(), value.toString(), Field.Store.YES, Field.Index.ANALYZED));
                
                // add literal props to the text field
                if (addToPrefText) {
                    addToPrefText(doc, value);
                }
                addToText(doc, value);
            }
        }
        
        // index each of the object properties
        for (EIURI propURI: result.getObjectProperties()) {
            final Set<EIURI> values = result.getObjectProperty(propURI);
            for (EIURI value: values) {
                // index the property value using the URI
                doc.add(new Field(propURI.toString() + OBJECT_URI_POSTFIX, value.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
                
                // add it to the related field
                // TODO boost high-value properties
                Field field = new Field(RELATED, value.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED);
                doc.add(field);   
                field.setBoost(MEDIUM_BOOST);
            }            
        }
    }
    
    /*
     * Index the types (potentially with materialization)
     */
    private void indexTypes(final Document doc, final EIURI typeURI, final boolean materializeTypes, final boolean indexURIs) {
        for (EIClass type : AbstractGenerator.getTypes(eagleiOntModel, typeURI, materializeTypes)) {
            final String uri = type.getEntity().getURI().toString();
            // index the inferred URIs
            if (indexURIs) {
                final Field field = new Field(INFERRED_TYPE, uri, Field.Store.YES, Field.Index.NOT_ANALYZED);
                doc.add(field);
                if (uri.equals(typeURI.toString())) {
                    field.setBoost(HIGHEST_BOOST);
                } else {
                    field.setBoost(HIGH_BOOST);
                }
            }
            final String label = eagleiOntModel.getPreferredLabel(type.getEntity().getURI());
            if (label != null) {
                if (uri.equals(typeURI.toString())) {
                    // add the label to the pref_text if the direct type
                    addToPrefText(doc, label);
                } else {
                    // store label using rdf:type if not direct type (direct is added using standard object prop
                    // logic)
                    doc.add(new Field(RDF.type.getURI(), label, Field.Store.YES, Field.Index.ANALYZED));                    
                }
                // add both direct and inferred to text
                addToText(doc, label);
            }
        }
    }
    
    /*
     * Adds the specified string value to the document's text field 
     */
    private static void addToText(final Document doc, final String value) {
        final Field field = new Field(TEXT, value, Field.Store.YES, Field.Index.ANALYZED);
        field.setBoost(STANDARD_BOOST);
        doc.add(field);
    }

    /*
     * Adds the specified string value to the document's pref_text field 
     */
    private static void addToPrefText(final Document doc, final String value) {
        final Field field = new Field(PREF_TEXT, value, Field.Store.YES, Field.Index.ANALYZED);
        field.setBoost(HIGH_BOOST);
        doc.add(field);
    }
    
    public void updateIndex() throws IOException {
 
        // execute a query for all resources
        final SearchRequest request = new SearchRequest();
        request.setMaxResults(MAX_RESULTS_TO_INDEX_FROM_EACH_NODE); 
        final SearchResultSet resultSet = this.nestedProvider.query(request);
        if (resultSet.getTotalCount() == 0) {
        	return;
        }

        processIndexChangeEvent(new IndexChangeEvent(new ChangeEventPayloadImpl(resultSet)));
    }
    

	public void run() {

        while (true) {
            try {
                // (re)build the Lucene index based on a query against the underlying provider
                this.updateIndex();
            } catch (Throwable t) {
                logger.error("Failed to rebuild lucene index", t);
            }
            
            try {
                // sleep for the specified time period
                Thread.sleep(this.getUpdateFrequency());
            } catch (InterruptedException ie) {
                // swallow
            }
        }
    }    
    

    /**
     * Dynamically builds the Lucene index from results retrieved from a nested SearchProvider.
     * 
     * To support the indexing of indirect properties (e.g. A->B->"some literal", 
     * where A is the RDF resource associated with the Lucene Document), a 2-pass indexing scheme is performed:
     * 
     * <ul>
     * <li>During the first pass all Documents for the resources associated with all input SearchResults are created. If a Document already 
     * exists, it is deleted. For SearchResults that represent deleted resources, a Document is not added back. 
     * This phase corresponds to multiple calls to LuceneSearchProviderIndexer.indexSearchResult().
     * <li>During the second pass, the Documents are updated with indirect properties. The set of Documents to updated corresponds to all new
     * Documents plus the Documents for all referencing Resources. The update call is performed by LuceneSearchProviderIndexer.addIndirectProperties()
     * </ul>
     * @param changeResults
     * @throws IOException 
     */
		private void processPayload (ChangeEventPayload payload) throws IOException {
	    	List<ChangeEventPayloadItem> changeSet = payload.getChanges();
	    	if (changeSet == null) {
	    		return;
	    	}

	        // Remember the embedded ones
	        final Set<ChangeEventPayloadItem> embeddedResources = new HashSet<ChangeEventPayloadItem>();
	        if (payload.getTotalCount() > 0) {

	            // Phase 1 of indexing: delete all Documents from index corresponding to SearchResults and
	            // recreate if the SearchResult is not for a deleted resource or an embedded resource.
	            
	            // index all of the results
	            for (ChangeEventPayloadItem change : changeSet) {

	            	if (!embedded(change)) {
		                indexSearchResult(change, true);
	            	} else {
	            		// Remember so we don't have to call the embedded method again
	            		embeddedResources.add(change);
	            	}
	            }
	            
	            // commit the index changes
	            commit();
	            
	            // Phase 2 of indexing: index the indirect properties of the new SearchResults and all referencing 
	            // search results
	                    
	            if (LuceneSearchProviderIndexer.INDEX_OBJECT_PROP_LABELS) {
	                // build up a list of documents that require an update due
	                // to referenced data (labels of object properties)
	                final Set<EIURI> docsToUpdate = new HashSet<EIURI>();
	                for (ChangeEventPayloadItem change : changeSet) {
	                    EIURI uri = change.getEntity().getURI();
	                    // by default, we are going to update (to index the labels
	                    // of object properties) all docs we originally indexed 
	                    // (unless they are deleted docs or embedded)
	                    if (!LuceneSearchProviderIndexer.isDeletedSearchResult(change) &&
	                        !embeddedResources.contains(change)) {
	                        docsToUpdate.add(uri);
	                    }

	                    // search for all docs that reference this doc
	                    // want to update each of them.
	                    List<EIURI> relatedDocs = getRelatedDocuments(uri);
	                    // If this doc is embedded, add its properties to the embedding
	                    // docs (the docs that reference this doc)
	                    if (embeddedResources.contains(change)) {
	                    	processEmbeddedResources(change, relatedDocs);
	                    }
		                docsToUpdate.addAll(relatedDocs);
	                }

	             // commit the index changes to embedding docs
	                commit();    
	                
	                // reindex all of the results with object prop labels
	                for (EIURI uri: docsToUpdate) {
	                    addIndirectProperties(uri);
	                }

	                // commit the index changes
	                commit();
	            }
	        }
	    }

	    /**
	     * Determines if the given SearchResult is an embedded resource. Embedded resources
	     * are not indexed separately; they are indexed as part of the
	     * embedding resource.
	     * 
	     * A resource may be embedded itself (if it is annotated as being embedded),
	     * or it may be a subtype of an embedded resource. 
	     * 
	     * @param result SearchResult
	     * @return TRUE if the result is embedded
	     */
	    
	    private boolean embedded(ChangeEventPayloadItem result) {
	    	 final EIEntity entityType = result.getType();
	         final EIURI typeURI = entityType.getURI();
	         return embedded(typeURI);
		}
	    
	    /**
	     * Same as embedded(SearchResult), but takes an EIURI as an argument.
	     * 
	     * @param typeURI EIURI
	     * @return TRUE If the resource is embedded
	     */
	    private boolean embedded(EIURI typeURI) {
	        // Note: The following statements do not work because the 
	        // annotations in EIClass are lazily instantiated. 
	        // "notes" will always be null. We have to use
	        // getClassAnnotations to populate the annotations first.
	        //
	        //EIClass entityClass = eagleiOntModel.getClass(typeURI);
	        //Set<String> notes = entityClass.getAnnotations();

	        final Set<String> notes = eagleiOntModel.getClassAnnotations(typeURI);
	        if (notes == null) {
				return false;
	        }
	              
	        // See if the resource has the embedded annotation.
	        if (notes.contains(EMBEDDED_CLASS)) {
	       	 return true;
	        }
	        // Check if the resource is a subtype of an embedded resource.
	        return embeddedSuperclass(typeURI);
	    }
	    
	    /**
	     * Determines if the given EIURI is an embedded resource
	     * by checking its superclasses. 
	     * 
	     * @param typeURI EIURI
	     * @return TRUE if the resource is embedded
	     */
	    private boolean embeddedSuperclass (EIURI typeURI) {
	        // Note: The getSuperClasses method of EIClass will not
	        // work because the list of superclasses is lazily instantiated.        
	        final List<EIClass> superClasses = eagleiOntModel.getSuperClasses(typeURI);
	        if (superClasses == null) {
	           return false;
	        }
	        for (EIClass eiClass : superClasses) {
	        	 EIEntity entity = eiClass.getEntity();
	        	 EIURI uri = entity.getURI();
	        	 // Special hard-coding of Mutation Type as an "embedded" class
	        	 // TODO: Post-1.1-MS5 there will be an annotation for this use case
	        	 if (uri.equals("http://www.berkeleybop.org/ontologies/owl/SO#SO_0001059")) {
	        	     return true;
	        	 }
		    	 final Set<String> notes = eagleiOntModel.getClassAnnotations(uri);//
		         if (notes == null) {
					return false;
		         }
		         // See if the resource has the embedded annotation.
		         if (notes.contains(EMBEDDED_CLASS)) {
		        	 return true;
		         }
		         // If one superclass is embedded, so is this one.
		         if (embeddedSuperclass(uri)) {
		        	 return true;
		         }
		    }
	        // None of the superclasses is embedded.
	        return false;
	    }
	     
	    /**
	     * Process an embedded resource by adding all of its properties to 
	     * any document that refers to it (embeds it).
	     * 
	     * @param embedded SearchResult Contains the embedded resource
	     * @param List<EIURI> embeddingDocs The List of documents that refer to the embedded resource. 
	     * @throws IOException 
	     */
	    private void processEmbeddedResources(ChangeEventPayloadItem embedded, List<EIURI> embeddingDocs) throws IOException {
	    	
	    	// Don't add the label and type properties to the embedding docs
	  //  	EIEntity label = embedded.getEntity();
	  //  	EIEntity type = embedded.getType();
			// Add each property of the embedded doc (result) to each of the embedding resources.
			for (EIURI embeddingDocURI : embeddingDocs) {
				final Document embeddingDoc = getDocumentByURI(embeddingDocURI);
				
				// Add all properties of the embedded doc to the embedding doc
				indexProperties(embedded, embeddingDoc);
				updateDocument(embeddingDocURI, embeddingDoc);
			}
	    }
			
		    	
	    /**
	     * Tedstub
	     */
	    public void processIndexChangeEvent_X(IndexChangeEvent e) throws IOException {
	        if (e == null) {
	            // could assert
	            return;
	        }
	        
	        ChangeEventPayload payload = e.getPayload();
	        List<ChangeEventPayloadItem> changeSet = payload.getChanges();
	        if (changeSet == null) {
                // could assert
	            return;
	        }
	        // index all of the changed resources
	        for (ChangeEventPayloadItem change : changeSet) {
	            processResourceChange_X(change);
	        }
	    }
	    
	    /**
	     * Tedstub
	     */
	    private void processResourceChange_X(ChangeEventPayloadItem resource) {
	        try {
                boolean isDelete = isDeletedSearchResult(resource);
    	        boolean isEmbedded = embedded(resource);
                EIURI uri = resource.getEntity().getURI();
                // compute docs that reference this doc
                List<EIURI> relatedDocs = getRelatedDocuments(uri);
                
                // ---  Default processing ---
                if (!isEmbedded) {
                    // This deletes the resource, will re-add if this is a change op.
                    indexSearchResult(resource, true);
                    commit();        
                }
    	        
               // ---  Embedded resource processing ---
               // If this doc is embedded, add its properties to the embedding
               // docs (the docs that reference this doc, should only be one...)
               if (!isDelete && isEmbedded) {
                   processEmbeddedResources(resource, relatedDocs);
                   commit();        
               }
                
               // ---  Related resource processing ---
               if (!isDelete && !isEmbedded) {
                   // Index object prop labels for the changed resource.
                   // TODO it's kind of weird that this logic is here and not in the default
                   //      indexing of the document.  Can/Should this be moved?
                   addIndirectProperties(resource.getEntity().getURI());
               }
               // Re-index resources that reference the changed resource.
               // TODO Should be skipping this if the resource is an embedded resource?
               for (EIURI relatedURI: relatedDocs) {
                   addIndirectProperties(relatedURI);
               }
               commit();
                
	        } catch (IOException ex) {
	            logger.error("Error indexing resource: " + resource.getEntity(), ex);
	        }
	    }	    
    
}
