package org.eaglei.search.provider.lucene.harvest;

import java.io.BufferedReader;
import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashSet;
import java.util.List;
import java.util.Locale;
import java.util.Set;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIDatatypeProperty;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.harvest.PollingDataHarvester;
import org.eaglei.search.harvest.ResourceChangeEvent;
import org.eaglei.search.harvest.ResourceChangeListener;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.ProviderUtils;
import org.eaglei.services.repository.RepositoryHttpConfig;

import com.hp.hpl.jena.ontology.DatatypeProperty;
import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.query.ResultSetFormatter;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.rdf.model.ModelFactory;
import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.rdf.model.RDFNode;
import com.hp.hpl.jena.rdf.model.Resource;
import com.hp.hpl.jena.rdf.model.Statement;
import com.hp.hpl.jena.sparql.resultset.SPARQLResult;
import com.hp.hpl.jena.sparql.resultset.XMLInput;
import com.hp.hpl.jena.vocabulary.RDF;
import com.hp.hpl.jena.vocabulary.RDFS;

/**
 * <ul>
 * <li>The results from the harvest call (returned in SPARQL tabular format) are parsed into a set of SPARQL 
 *     QuerySolutions.
 * <li>The QuerySolutions are used to create a set of ResourceChangeEvents (one event for each unique subject URI). 
 * <li>ResourceChangeListeners are notified when a set of change notifications are about to being, 
 *     on each event, and on change stream end.
 * </ul>
 * 
 * @author tbashor
 */
public abstract class AbstractStreamHarvester implements PollingDataHarvester {
    
    private class DataHarvestPoller extends Thread {
        
        DataHarvestPoller() {
            super("DataHarvestPoller");
            setPriority(Thread.MIN_PRIORITY);
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
                try {
                    harvest();
                } catch (Throwable t) {
                    logger.error("Unexpected error during harvest", t);
                }
        		isInitialPollComplete = true;
                
                try {
                    //TODO make configurable
                    Thread.sleep(10000);
                } catch (InterruptedException ie) {
                    // swallow
                }
            }
        }
        
    }

    private static final Log logger = LogFactory.getLog(AbstractStreamHarvester.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    private static final String SUBJECT= "subject";
    private static final String PREDICATE= "predicate";            
    private static final String OBJECT = "object";

    protected static final DateFormat formatWithTZ = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS z");
    private static final DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    private static final SimpleDateFormat preciseHTTPDate =
        new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss.SSS zzz");
    static {
        format.setTimeZone(TimeZone.getTimeZone("GMT"));
        preciseHTTPDate.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "GMT"), Locale.getDefault()));
    }
    
    protected final EIOntModel eiOntModel;
    protected final EIEntity institution;
    
    private final List<ResourceChangeListener> listeners = new ArrayList<ResourceChangeListener>();
    private final Set<String> labelURIs;
    private final Set<EIURI> resourceProviderProperties;
    
    protected boolean isInitialized;
    private boolean isInitialPollComplete = false;
    private DataHarvestPoller pollingThread;
        
    /**
     * Creates a new RepositoryHarvester.
     */
    protected AbstractStreamHarvester(final EIOntModel eiOntModel, final EIEntity institution) {
        this.eiOntModel = eiOntModel;
        this.institution = institution;
  
        List<Property> preferredLabelProperties = ((JenaEIOntModel) eiOntModel).getPrefLabelProperties();
        labelURIs = new HashSet<String>();
        for (Property prop: preferredLabelProperties) {
            labelURIs.add(prop.getURI());
        }
        
        // Assumes that these properties are ONLY used as resource provider
        // relationships.  This might not be a good thing, since they are being
        // automatically filtered out of the object property list.
        resourceProviderProperties = eiOntModel.getResourceProviderProperties();
        
        setInitialized(false);
    }
    
    @Override 
    public boolean isInitialPollComplete() {
    	return isInitialPollComplete;
    }
    
    @Override
    public synchronized void startPolling() {
        if (pollingThread != null) {
            return;
        }
        pollingThread = new DataHarvestPoller();
        pollingThread.start();
    }
    
    public boolean isInitialized() {
        return isInitialized;
    }
    
    public void setInitialized(boolean isInitialized) {
        this.isInitialized = isInitialized;
    }
    
    public abstract String getHarvestInfo();
    
    public void addChangeListener(ResourceChangeListener listener) {
    	listeners.add(listener);
    }
    
    protected void notifyChangeStreamStart() {
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeStreamStart(this.institution);
    	}
    }

    protected void notifyChangeEvent(ResourceChangeEvent event) {
    	if (event == null) {
    		return;
    	}
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeEvent(event);
    	}
    }

    protected void notifyChangeStreamEnd(Date lastModifiedDate) {
    	for (ResourceChangeListener listener : listeners) {
    		listener.onChangeStreamEnd(this.institution, lastModifiedDate);
    	}
    }
    
    protected int generateResourceChangeEvents(InputStream is) {
        int count = 0;
        String currentURI = null;
        Model model = null;
        final ResultSet resultSet = ResultSetFactory.fromXML(is);
        for ( ; resultSet.hasNext() ; )
        {
            QuerySolution soln = resultSet.next() ;
            final Resource resource = soln.getResource(SUBJECT);
            String nextURI = resource.getURI();
            if (currentURI == null) {
                // First data read
                currentURI = nextURI;
                model = ModelFactory.createDefaultModel();
            } else if (! currentURI.equals(nextURI)) {
                // New resource data, send data for previous resource
                notifyChangeEvent(createChangeEventFromModel(currentURI, model));
                count++;
                currentURI = nextURI;
                model = ModelFactory.createDefaultModel();
            }
            final Resource property = soln.getResource(PREDICATE);
            final RDFNode value = soln.get(OBJECT);
            final Property predicate = model.createProperty(property.getURI());
            model.add(model.createStatement(resource, predicate, value));
        }
        if (currentURI != null) {
            // All data read
            notifyChangeEvent(createChangeEventFromModel(currentURI, model));
            count++;
        }
        return count;
    }
        
    private ResourceChangeEvent createChangeEventFromModel(final String uri, final Model model) {
    	ResourceChangeEvent result;      
        final Resource resource = model.getResource(uri);
 
        Property isDeleted = model.getProperty(EIOntConstants.IS_DELETED);
        if (isDeleted != null) {
            Statement s = resource.getProperty(isDeleted);
            if (s != null && s.getObject().isLiteral()) {
                if (((Literal) s.getObject()).getBoolean()) {
                    // has the isDeleted property and the value is true
                    final EIEntity entity = 
                    	EIEntity.create(EIURI.create(resource.getURI()), resource.getLocalName());
                	result = new ResourceChangeEvent(EIOntConstants.IS_DELETED, entity);
                    result.setEntity(entity);
                    result.setInstitution(institution);
                    return result;
                }
            }
        }

        // Do we need to check other potential resource label properties?
        final Statement labelStmt = resource.getProperty(RDFS.label);
        String label = null;
        if (labelStmt != null) {
            label = labelStmt.getLiteral().getLexicalForm();
        }
        if (label == null) {
            logger.error("Resource " + resource.getURI() + " is missing label");
            return null;
        }
        EIEntity entity = 
        	EIEntity.create(EIURI.create(resource.getURI()), label);

        result = new ResourceChangeEvent(null, entity);
        // TODO should confirm that this result is actually associated with this institution
        result.setInstitution(institution);

        // gets the first asserted type in the model
        // TODO handle multiple asserted types
        final Statement type = resource.getProperty(RDF.type);
        String typeURIStr = null;
        if (type != null) {
            typeURIStr = ((Resource)type.getObject()).getURI();
        }
        if (typeURIStr == null) {
            logger.error(entity.toString() + " is missing type");
            return null;
        }
        final EIURI typeURI = EIURI.create(typeURIStr);
        if (!eiOntModel.isModelClassURI(typeURIStr)) {
            logger.error(entity.toString() + " has type " + typeURIStr + " that does not exist in eagle-i data model");
            return null;
        }
        EIClass typeClass = eiOntModel.getClass(typeURI);
        assert (typeClass != null);
        final EIEntity typeEntity = typeClass.getEntity();
        result.setType(typeEntity);
        
        for (EIProperty eiProp : eiOntModel.getProperties(typeURI)) {
        	final EIURI eiURI = eiProp.getEntity().getURI();
            final Property modelProp = model.getProperty(eiURI.toString());
            if (modelProp == null) {
            	// property isn't in the resource instance
            	continue;
            }
            for (RDFNode object : model.listObjectsOfProperty(modelProp).toList()) {
	        	if (eiProp instanceof EIDatatypeProperty) {
	        		if (!(object instanceof Literal)) {
	                    logger.error(entity.toString() + " of type " + typeEntity.toString() + " has datatype property " + eiProp.toString() + " with non-Literal value: " + object.toString());
	        			continue;
	        		}
	                result.addDataTypeProperty((EIDatatypeProperty)eiProp, ((Literal) object).getLexicalForm());
	        	} else {
	        		if (!(object instanceof Resource)) {
	                    logger.error(entity.toString() + " of type " + typeEntity.toString() + " has object property " + eiProp.toString() + " with non-Resource value: " + object.toString());
	        			continue;
	        		}
	                // if the property is a resource, get the URI
	                final Resource value = (Resource) object;
	                // save the entity as a result property
	                EIURI valueURI = EIURI.create(value.getURI());
	                if (resourceProviderProperties.contains(eiURI)) {
	                    // Assumes that these properties are ONLY used as resource provider
	                    // relationships.  This might not be a good thing, since they are being
	                    // automatically filtered out of the object property list.
	                    result.setProvider(valueURI);
	                } else {
		                result.addObjectProperty((EIObjectProperty)eiProp, valueURI);	                	
	                }
	        	}
        	}
        }
        
        return result;
    }

}