package org.eaglei.search.provider.lucene.search;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.TreeSet;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.queryParser.ParseException;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.TopFieldDocs;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.QueryScorer;
import org.apache.lucene.store.Directory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.JenaEIOntModel;
import org.eaglei.search.provider.ClassCountResult;
import org.eaglei.search.provider.CountResult;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.services.InstitutionRegistry;

import com.hp.hpl.jena.rdf.model.Property;
import com.hp.hpl.jena.vocabulary.RDF;

/**
 * SearchProvider that queries a Lucene index populated according to the schema
 * defined in LuceneSearchIndexSchema.
 * 
 * @author frost
 */
public final class LuceneSearchProvider implements LuceneSearchIndexSchema, SearchProvider {

	private static final Log logger = LogFactory
			.getLog(LuceneSearchProvider.class);
	private static final boolean DEBUG = logger.isDebugEnabled();

	/*
	 * Controls whether datatype and object properties are returned.
	 */
	private static final boolean RETRIEVE_PROPERTIES = false;

	/*
	 * Handle to the in-memory eagle-i ontology
	 */
	private final EIOntModel eagleiOntModel;
	private final InstitutionRegistry institutionRegistry;
	/*
	 * The Lucene index. This may be in-memory or file-based.
	 */
	private Directory dir;
	/*
	 * The Lucene analyzer used for both indexing and query execution.
	 */
	private Analyzer analyzer;
	/**
	 * Helper class that creates a Query from a SearchRequest
	 */
	private LuceneSearchQueryBuilder queryBuilder;

	private ClassUsageCache classUsageCache;

	/**
	 * Creates a LuceneProvider that executes SearchRequests over the specified
	 * Directory using the specified Analyzer. The Directory must be populated
	 * using LuceneIndexer.
	 * 
	 * @param eagleiOntModel
	 *            Reference to the eagle-i ontology model.
	 * @param dir
	 *            Directory holding the Lucene index.
	 * @param analyzer
	 *            Analyzer to use for query execution.
	 */
	public LuceneSearchProvider(final EIOntModel eagleiOntModel, final InstitutionRegistry institutionRegistry,
			final Directory dir, final Analyzer analyzer, 
			final LuceneSearchQueryBuilder queryBuilder, final ClassUsageCache classUsageCache) {
		assert dir != null;
		assert analyzer != null;
		this.eagleiOntModel = eagleiOntModel;
		this.institutionRegistry = institutionRegistry;
		this.dir = dir;
		this.analyzer = analyzer;
		this.queryBuilder = queryBuilder;
		this.classUsageCache = classUsageCache;
	}

	@Override
	public void init() throws IOException {
		// no-op
	}

	/*
	 * (non-Javadoc)
	 * 
	 * @see
	 * org.eaglei.search.provider.SearchProvider#query(org.eaglei.search.request
	 * .SearchRequest)
	 */
	public SearchResultSet query(final SearchRequest request)
			throws IOException {
		return query(request, true);
	}

	/*
	 * Version of query that allows the optional creation of search results.
	 * This is supported for the count use case (when retrieving counts,
	 * creation of actual results is not necessary).
	 * 
	 * @param request The search request
	 * 
	 * @param createResults True to create SearchResult objects. False to just
	 * execute the query and get counts.
	 */
	private SearchResultSet query(final SearchRequest request,
			final boolean createResults) throws IOException {
		assert request != null;

		// create the query
		Query query = null;
		try {
			query = this.queryBuilder.createQuery(request);
	        if (DEBUG && createResults) {
	            logger.debug("Query: " + query.toString());
	        }
		} catch (ParseException pe) {
			throw new IOException(pe.getLocalizedMessage());
		}

		// if there was an error creating the query, just return an empty result
		// set
		if (query == null) {
			return new SearchResultSet(request);
		}

		// execute the search
		return executeSearch(request, query, createResults);
	}

	/*
	 * Executes the Lucene query and creates SearchResults from the output docs.
	 * 
	 * @param request The SearchRequest
	 * 
	 * @param query The Lucene Query
	 * 
	 * @param createResults True if results should be created. False to just get
	 * a count.
	 */
	private SearchResultSet executeSearch(final SearchRequest request,
			final Query query, final boolean createResults) throws IOException {

		final SearchResultSet results = new SearchResultSet(request);

		// create an IndexSearcher
		// TODO would like to create and reuse a single IndexSearcher but it is
		// not seeing index changes on
		// reopen...
		final IndexSearcher searcher = new IndexSearcher(this.dir, true);
		searcher.setDefaultFieldSortScoring(true, true);

		// Execute the Query using the IndexSearcher
		final TopFieldDocs docs = searcher.search(query, null, request
				.getStartIndex()
				+ request.getMaxResults(), Sort.RELEVANCE);

		// if (DEBUG) {
		// logger.debug("Found " + docs.totalHits + " matches");
		// }

		// Set the count data in the SearchResultSet
		results.setTotalCount(docs.totalHits);
		results.setStartIndex(request.getStartIndex());

		// if not populating individual SearchResults, return with the counts
		if (!createResults) {
			return results;
		}

		// create a highligher for the query
		final Highlighter highlighter = new Highlighter(new QueryScorer(query));

		// get page subset and create a SearchResult for each Document in the
		// page
		final int cap = request.getStartIndex() + request.getMaxResults();
		for (int i = request.getStartIndex(); (i < cap)
				&& (i < docs.scoreDocs.length); i++) {

			// get the Lucene ScoreDoc and Document
			final ScoreDoc scoreDoc = docs.scoreDocs[i];
			final Document document = searcher.doc(scoreDoc.doc);

			// create a SearchResult from the document fields
			final SearchResult result = createSearchResultForDocument(searcher,
					scoreDoc, document);

			// if there was an issue creating the result, skip
			if (result == null) {
				continue;
			}

			// compute the highlight
			final String highlight = LuceneSearchUtil.computeHighlight(
					highlighter, analyzer, eagleiOntModel, request, query,
					document, result);
			if (highlight != null) {
				result.setHighlight(highlight);
			}

			// check for dups in the resultset
			if (results.getResults().contains(result)) {
				logger.error("Found duplicate result");
			}

			// add the result
			results.getResults().add(result);
		}

		return results;
	}

	/**
	 * Creates a SearchResult for a single Lucene Document match
	 * 
	 * @param searcher
	 *            IndexSearcher used to execute the query.
	 * @param scoreDoc
	 *            The ScoreDoc for the match
	 * @param document
	 *            The Document for the match
	 * @return The SearchResult
	 */
	private SearchResult createSearchResultForDocument(
			final IndexSearcher searcher, final ScoreDoc scoreDoc,
			final Document document) {

		// get the document score
		final float score = scoreDoc.score;

		// get the resource URI
		final String resource = document.get(FIELD_URI);
		// get the resource label
		String label = document.get(FIELD_ENTITY_LABEL);
		if (label == null) {
			// A very exceptional case.  We hope.
			// A stub resource was committed to the index.
			logger.warn("A stub resource instance w/ no label was included as a search result: " + resource);
			return null;
		}
		// create an EIEntity for the resource
		final EIEntity resourceEntity = EIEntity.create(EIURI.create(resource),
				label);

		// get the EIInstitution
		final String institutionURI = document.get(FIELD_INSTITUTION_URI);
		final EIEntity institutionEntity = institutionRegistry.getInstitution(EIURI.create(institutionURI));

		// get the type
		final String typeURI = document.get(FIELD_ASSERTED_TYPE_URI);
		final String typeLabel = document.get(FIELD_ASSERTED_TYPE_LABEL);
		final EIEntity typeEntity = EIEntity.create(typeURI, typeLabel);

		// create the SearchResult
		final SearchResult result = new SearchResult(resourceEntity, typeEntity, null, institutionEntity);
		
		setResourceProvider(result, document);

		// add all datatype and object properties
		/*
		for (Fieldable f : document.getFields()) {
			addResultProperty(result, document, f);
		}
		*/

		result.setURL(resource);
		result.setRank(score);

		return result;
	}
	
	private void setResourceProvider(SearchResult result, Document doc) {
		final String providerURI = doc.get(FIELD_PROVIDER_URI);
		final String providerLabel = doc.get(FIELD_PROVIDER_LABEL);
		if (providerURI != null && providerLabel != null) {
			EIEntity providerEntity = EIEntity.create(providerURI, providerLabel);
			result.setProvider(providerEntity);
		}
	}


	@Override
	public SearchCounts count(SearchCountRequest request) throws IOException {
		throw new RuntimeException("deprecated");
	}

	@Override
	public ClassCountResult getResourceCount(SearchRequest request) {
		ClassCountResult totalCountResult = classUsageCache
				.getClassCount(request);
		if (request.getTerm() == null
				&& request.getLocation() == null
				&& (request.getBinding() == null || (request.getBinding()
						.getDataTypeProperties().size() == 0 && request
						.getBinding().getObjectProperties().size() == 0))) {
			return totalCountResult;
		}
		SearchResultSet results;
		ClassCountResult result = new ClassCountResult(request);
		result.setClassCount(new CountResult(totalCountResult.getEntity()));
		try {
			// pass in false to so we don't create results
			results = query(request, false);
			result.getClassCount().increment(results.getTotalCount());
			SearchRequest countRequest = new SearchRequest(request
					.toURLParams());
			if (totalCountResult.getSuperClassCounts() != null) {
				List<CountResult> resultSuperClassCounts = new ArrayList<CountResult>(
						totalCountResult.getSuperClassCounts().size());
				for (CountResult totalSuperClassCount : totalCountResult
						.getSuperClassCounts()) {
					countRequest.setBinding(new SearchRequest.TypeBinding(
							totalSuperClassCount.getEntity().getURI()));
					results = query(countRequest, false);
					resultSuperClassCounts.add(new CountResult(
							totalSuperClassCount.getEntity(), results
									.getTotalCount()));
				}
				result.setSuperClassCounts(resultSuperClassCounts);
			}
			if (totalCountResult.getSubClassCounts() != null) {
				TreeSet<CountResult> resultSubClassCounts = new TreeSet<CountResult>();
				for (CountResult totalSubClassCount : totalCountResult
						.getSubClassCounts()) {
					countRequest.setBinding(new SearchRequest.TypeBinding(
							totalSubClassCount.getEntity().getURI()));
					results = query(countRequest, false);
					if (results.getTotalCount() > 0) {
						resultSubClassCounts.add(new CountResult(
								totalSubClassCount.getEntity(), results
										.getTotalCount()));
					}
				}
				result.setSubClassCounts(resultSubClassCounts);
			}
		} catch (IOException e) {
			logger.error(e);
		}
		return result;
	}

}
