package org.eaglei.search.provider.lucene.search;

import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Fieldable;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.highlight.Highlighter;
import org.apache.lucene.search.highlight.InvalidTokenOffsetsException;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;

public class LuceneSearchUtil {
	
    private static final Log logger = LogFactory.getLog(LuceneSearchUtil.class);
    
	static String DATATYPE_PROP_FIELD_PREFIX = "DTPropValue_";
	static String OBJECT_PROP_FIELD_PREFIX = "OBPropValue_";
	static String VALUE_URI_PREFIX = "#URI_";

	public static String getPropertyFieldName(EIEntity propEntity, String valueURIStr) {
		StringBuilder buf = new StringBuilder();
		if (valueURIStr != null) {
			buf.append(OBJECT_PROP_FIELD_PREFIX);
		} else {
			buf.append(DATATYPE_PROP_FIELD_PREFIX);
		}
		buf.append(propEntity.getLabel().replace(' ', '_'));
		if (valueURIStr != null) {
			buf.append(VALUE_URI_PREFIX);
			buf.append(valueURIStr);
		}
		return buf.toString();
	}

	public static String getPropertyLabel(String fieldName) {
		if (fieldName.startsWith(DATATYPE_PROP_FIELD_PREFIX)) {
			return fieldName.substring(DATATYPE_PROP_FIELD_PREFIX.length()).replace('_', ' ');
		} else {
			int labelEnd = fieldName.indexOf(VALUE_URI_PREFIX);
			return fieldName.substring(OBJECT_PROP_FIELD_PREFIX.length(), labelEnd).replace('_', ' ');
		}
	}
	
	public static String getPropertyValueURIString(String fieldName) {
		if (!fieldName.startsWith(OBJECT_PROP_FIELD_PREFIX)) {
			return null;
		}
		int labelEnd = fieldName.indexOf(VALUE_URI_PREFIX);
		if (labelEnd == -1) {
			return null;
		}
		return fieldName.substring(labelEnd + VALUE_URI_PREFIX.length());
	}
	
    /*
     * Maximum number of highlight field matches to return
     */
    private static final int MAX_HIGHLIGHT_MATCHES = 3;
    
    /**
     * Computes the highlight for a search request.
     * @param highlighter The highlighter to use for computation
     * @param analyzer The Lucene analyzer used to generate the match
     * @param ontModel Reference to the eagle-i ontology model.
     * @param request The search request
     * @param query The Lucene query
     * @param document Matching document
     * @param result The SearchResult
     * @return Computed highlight
     */
    protected static String computeHighlight(final Highlighter highlighter, final Analyzer analyzer, final EIOntModel ontModel, final SearchRequest request, final Query query, final Document document, final SearchResult result) {
        // compute a highlight if this was a term query or had a datatype binding
        SearchRequest.Term term = request.getTerm();
        SearchRequest.TypeBinding binding = request.getBinding();
        if ((term != null && term.getQuery() != null) || (binding != null && !binding.getDataTypeProperties().isEmpty())) {
            // if we can locate the associated property & label, prepend that
            // TODO, try and re-enable, encountering performance problems
            final StringBuilder highlights = new StringBuilder();
            boolean firstHighlight = true;
            int totalHighlightMatches = 0;
            for (Fieldable f: document.getFields()) {
                final String name = f.name();
                final String strValue = f.stringValue();
                if (f.isTokenized() && 
                		(name.startsWith(LuceneSearchUtil.DATATYPE_PROP_FIELD_PREFIX) || 
                				name.startsWith(LuceneSearchUtil.OBJECT_PROP_FIELD_PREFIX))) {
                    String highlight = getHighlightForField(highlighter, analyzer, name, strValue);
                    if (highlight != null) {
                        if (firstHighlight) {
                            highlights.append("...");
                            firstHighlight = false;
                        }
                        // optionally prepend the property label
                        final String propLabel = getPropertyLabel(name);
                        if (propLabel != null) {
                            highlights.append(propLabel + ": ");
                        }  else {
                            //logger.error("Could not find label for " + name);
                        }
                        highlights.append(highlight);
                        highlights.append("...");
                        
                        // check if we've the max allowed field matches
                        if (totalHighlightMatches++ >= MAX_HIGHLIGHT_MATCHES) {
                            break;
                        }
                    }
                }
            }
            return highlights.toString();
        }
        
        return null;
    }
    
    /*
     * Computes the match highlight for the specified field and field value.
     * @param highlighter The Lucene highlighter to use for highlight computation
     * @param fieldName Name of the index field
     * @param field value Value of the field
     * @return Computed highlight.
     */
    private static String getHighlightForField(final Highlighter highlighter, final Analyzer analyzer, final String fieldName, final String fieldValue) {
        try {
            final String highlight = highlighter.getBestFragment(analyzer, fieldName, fieldValue); //MAX_FRAGMENTS);
            if (highlight != null) {
                return highlight;
            }
        } catch (InvalidTokenOffsetsException itoe) {
            logger.error(itoe);
        } catch (IOException ioe) {
            logger.error(ioe);
        }
        return null;
    }
    
	
}
