package org.eaglei.search.provider.lucene.suggest;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.PhraseQuery;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.ResourceChangeEvent;
import org.eaglei.search.harvest.ResourceChangeListener;
import org.eaglei.search.provider.lucene.AbstractLuceneIndexer;

public class LuceneAutoSuggestIndexer extends AbstractLuceneIndexer implements LuceneAutoSuggestIndexSchema,ResourceChangeListener {

    private static final Log logger = LogFactory.getLog(LuceneAutoSuggestIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

    private Set<EIEntity> resourceRoots = new HashSet<EIEntity>();
    
    private static final EIURI RESOURCE_PROVIDER_PROPERTY = EIURI.create(EIOntConstants.PG_RELATED_RESOURCE_PROVIDER);
    
    public LuceneAutoSuggestIndexer(final EIOntModel eiOntModel, final Analyzer analyzer, final Directory directory) {
    	super(eiOntModel, analyzer, directory);
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
			resourceRoots.add(c.getEntity());
		}
    }
    
    @Override
    public void onChangeEvent(ResourceChangeEvent event) {
    	super.onChangeEvent(event);
    	if (event.isDelete()) {
    		deleteDocuments(event.getEntity().getURI());
    	} else {
    		indexResourceInstance(event);
    	}
    }

    @Override
    protected List<Document> getDocumentsFromIndex(final EIURI uri) {
    	try {
	        // create a query 
	        final PhraseQuery propQuery = new PhraseQuery();
	        propQuery.add(new Term(FIELD_URI, uri.toString()));
	        
	        final IndexSearcher searcher = new IndexSearcher(directory, true);
	        searcher.setDefaultFieldSortScoring(true, true);
	        
	        final TopDocs docs = searcher.search(propQuery, 1);
	        if (docs.totalHits == 0) {
	            //logger.error("Did not find " + uri + " in search index");
	            return null;
	        }
	        final List<Document> result = new ArrayList<Document>(docs.scoreDocs.length);
	        for (ScoreDoc scoreDoc : docs.scoreDocs) {
	        	result.add(searcher.doc(scoreDoc.doc));
	        }
	        return result;
    	} catch (IOException e) {
    		logger.error(e);
    		return null;
    	}
    }
    
    private List<Document> addStubInstanceDocuments(EIURI uri) {
    	//logger.debug("    Creating stub instance for " + uri);
    	List<Document> docs = new ArrayList<Document>(1);
        final Document doc = new Document();
        // flag that it's a resource
        doc.add(new Field(FIELD_IS_INSTANCE, Boolean.TRUE.toString(), Field.Store.YES, Field.Index.NO));
        // create a non-indexed field for the URI
        doc.add(new Field(FIELD_URI, uri.toString(), Field.Store.YES, Field.Index.NO));
        docs.add(doc);
        setDocuments(uri, docs);
        return docs;
    }
    
    /*
     * Note that this could be a new resource or an update.
     * If this is an update, the previous version of the Document
     * will be deleted on commit.
     */
    private List<Document> indexResourceInstance(final ResourceChangeEvent event) {
    	//logger.debug("Index resource " + event.getEntity() + "  Type: " + event.getType());

    	List<Document> docs;
    	Field[] isValueOfFields = null;
    	
    	// Check if this uri has already been encountered.
    	// Possibly as a object property value.
    	// Need to save off the referencing property list
    	docs = getDocuments(event.getEntity().getURI());
    	if (docs != null) {
    		// Assume that the IS_VALUE_OF field list is the same for all synonyms
    		isValueOfFields = docs.get(0).getFields(FIELD_IS_VALUE_OF);
    	}

        // TODO support alternate names
    	docs = new ArrayList<Document>(1);
        final Document doc = new Document();

        // flag that it's a resource
        doc.add(new Field(FIELD_IS_INSTANCE, Boolean.TRUE.toString(), Field.Store.YES, Field.Index.NO));
        // create a non-indexed field for the URI
        String uri = event.getEntity().getURI().toString();
        doc.add(new Field(FIELD_URI, uri, Field.Store.YES, Field.Index.NOT_ANALYZED));
        String label = event.getEntity().getLabel();
        doc.add(new Field(FIELD_ENTITY_LABEL, label, Field.Store.YES, Field.Index.NO));
        // create an indexed field with position offsets for computation of 
        // highlights
        doc.add(new Field(FIELD_LABEL, label, Field.Store.YES,
                Field.Index.ANALYZED, Field.TermVector.WITH_POSITIONS_OFFSETS));
        
        if (event.getInstitution() != null) {
            // create a non-indexed field for the providing institution
        	String institutionURI = event.getInstitution().getURI().toString();
        	doc.add(new Field(FIELD_INSTITUTION_URI, institutionURI, Field.Store.YES, Field.Index.NO));
        }
        
        // create type field
        setTypeField(true, doc, event.getType().getURI());
        
        if (isValueOfFields != null) {
        	// Add back any prior references
        	for (Field f : isValueOfFields) {
        		doc.add(f);
        	}
        }
        
        // Index the object property references.
        // Will create stub documents for any instance references
        // that we don't have data for yet.
        for (EIObjectProperty prop : event.getObjectProperties()) {
        	EIURI propURI = prop.getEntity().getURI();
        	for (EIURI valueURI : event.getObjectProperty(prop)) {
        		indexPropertyReference(valueURI, propURI);
        	}
        }
        
        // generate a special meta property for resource provider
        if (event.getProvider() != null) {
        	EIURI resourceProviderURI = event.getProvider();
        	indexPropertyReference(resourceProviderURI, RESOURCE_PROVIDER_PROPERTY);
        }
        
        docs.add(doc);
        setDocuments(event.getEntity().getURI(), docs);
        return docs;
    }
    
    private void indexPropertyReference(EIURI valueResourceURI, EIURI propertyURI) {
		// Note that referenced doc may be a type document or instance document.
		// The list of documents are for synonyms
		List<Document> referencedDocs = getDocuments(valueResourceURI);
		if (referencedDocs == null) {
	     	EIClass typeClass = eiOntModel.getClass(valueResourceURI);
	    	if (typeClass != null) {
	    		referencedDocs = addTypeDocuments(typeClass.getEntity().getURI());
	    	} else {
	    		// create stub instance URI
	    		referencedDocs = addStubInstanceDocuments(valueResourceURI);
	    	}
		}
		for (Document referencedDoc : referencedDocs) {
			boolean alreadyIndexed = false;
			for (String existingProp : referencedDoc.getValues(FIELD_IS_VALUE_OF)) {
				// Check if this property reference has already been indexed.
				if (existingProp.equals(propertyURI.toString())) {
					alreadyIndexed = true;
					break;
				}
			}
			if (alreadyIndexed) {
				// Assume that all docs for this valueURI will have the same
				// IS_VALUE_OF list.
				break;
			}
			referencedDoc.add(new Field(FIELD_IS_VALUE_OF, propertyURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED)); 
		}
    }

    /*
     * Populate the type field in a Document.  Same for both instances and type documents.
     */
    private void setTypeField(boolean isInstance, Document doc, EIURI typeURI) {
    	EIEntity rootType = null;
        //   the asserted type
        doc.add(new Field(FIELD_TYPE, typeURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        List<Document> docs = getDocuments(typeURI);
        if (docs == null) {
        	addTypeDocuments(typeURI);
        }
        if (rootType ==  null && isInstance) {
        	// If an instance, then just use the asserted type
        	EIClass type = eiOntModel.getClass(typeURI);
        	rootType = type.getEntity();
        }
        //   the super types
        List<EIClass> resourceSuperClasses = eiOntModel.getSuperClasses(typeURI);
        for (EIClass superClass : resourceSuperClasses) {
            EIURI superURI = superClass.getEntity().getURI();
            doc.add(new Field(FIELD_TYPE, superURI.toString(), Field.Store.YES, Field.Index.NOT_ANALYZED));
            List<Document> superDocs = getDocuments(superURI);
            if (superDocs == null) {
            	addTypeDocuments(superURI);
            }
            if (rootType == null && resourceRoots.contains(superClass.getEntity())) {
            	rootType = superClass.getEntity();
            }
        }
        if (rootType == null && resourceSuperClasses.size() > 0) {
        	// Must be an entity that is outside the resource heirarchy
        	rootType = resourceSuperClasses.get(resourceSuperClasses.size()-1).getEntity();
        }
        if (rootType != null) {
        	doc.add(new Field(FIELD_ROOT_TYPE, rootType.getLabel(), Field.Store.YES, Field.Index.NOT_ANALYZED));
        }
    }
    
    private List<Document> addTypeDocuments(EIURI typeURI) {
        // Note: need to put unpopulated doc in map immediately to short circuit
        // potential infinite recursion caused by calling setTypeField inside
        // createTypeDocuments()
    	List<Document> docs = new ArrayList<Document>();
        setDocuments(typeURI, docs);

        EIClass clazz = eiOntModel.getClass(typeURI);
        EIURI uri = clazz.getEntity().getURI();
        // Creates multiple Documents, one for each synonym.
        for (String label : eiOntModel.getLabels(uri)) {
	        Document doc = new Document();
	
	        // flag that it's not a resource
	        doc.add(new Field(FIELD_IS_INSTANCE, Boolean.FALSE.toString(), Field.Store.YES, Field.Index.NO));
	        // create a non-indexed field for the URI
	        doc.add(new Field(FIELD_URI, uri.toString(), Field.Store.YES, Field.Index.NO));
	        doc.add(new Field(FIELD_ENTITY_LABEL, clazz.getEntity().getLabel(), Field.Store.YES, Field.Index.NO));
	        // create an indexed field with position offsets for computation of 
	        // highlights
            doc.add(new Field(FIELD_LABEL, label, Field.Store.YES,
                    Field.Index.ANALYZED, Field.TermVector.WITH_POSITIONS_OFFSETS));
            // populate the type field
            setTypeField(false, doc, uri);
            docs.add(doc);
        }
        return docs;
    }
    
}
