package org.eaglei.search.provider.ncbi;

import java.io.IOException;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.ncbi.ESearch.ESearchResult;
import org.eaglei.search.provider.ncbi.ESummary.ESummaryResult;
import org.eaglei.services.InstitutionRegistry;

/**
 * Abstract implementation of the eagle-i SearchProvider interface that wraps an NCBIDBProvider.  
 *
 * @author rfrost
 */
public abstract class NCBISearchProvider implements SearchProvider {

    protected static final Log logger = LogFactory.getLog(NCBISearchProvider.class);
    protected static final boolean DEBUG = logger.isDebugEnabled();

    protected final NCBIDBProvider dbProvider;
    protected final EIOntModel eagleiOntModel;
    protected final InstitutionRegistry institutionRegistry;    

    /**
     * Creates a new NCBIDBProvider that executes queries against NCBI.
     */
    public NCBISearchProvider(final NCBIDBProvider dbProvider, final EIOntModel eagleiOntModel, final InstitutionRegistry registry) {
        assert dbProvider != null;
        assert eagleiOntModel != null;
        assert registry != null;
        this.dbProvider = dbProvider;
        this.eagleiOntModel = eagleiOntModel;
        this.institutionRegistry = registry;
    }
    
    public void init() throws IOException {
        // no-op
    }
    
    @Override
    public SearchResultSet query(SearchRequest request) throws IOException {
        final SearchResultSet resultSet = new SearchResultSet(request);
        if (request.getTerm() == null || request.getTerm().getQuery().length() == 0) {
        	return resultSet;
        }
        if (!shouldAddResults(request)) {
            return resultSet;
        }
        final String query = getQuery(request);
        final ESearchResult result = dbProvider.query(query, request.getStartIndex(), request.getMaxResults());
        resultSet.setTotalCount(result.count);
        resultSet.setStartIndex(request.getStartIndex());
        for (ESummaryResult sum: ESummary.parseESummaryResult(result.details)) {
        	final SearchResult searchResult = createSearchResultForDocSummary(sum);
            if (searchResult != null) {
                resultSet.getResults().add(searchResult);
            }
        }
        return resultSet;
    }        

    @Override
    public SearchCounts count(final SearchCountRequest countRequest) throws IOException {
        final SearchRequest request = countRequest.getRequest();
        final SearchCounts counts = new SearchCounts(request);
        for (EIURI type: countRequest.getCountTypes()) {
            final SearchRequest searchRequest = new SearchRequest(request.toURLParams());
            if (type == null) {
                searchRequest.setBinding(null); // get counts for all types
            } else {
                final SearchRequest.TypeBinding binding = request.getBinding();
                if (binding == null) {
                    searchRequest.setBinding(new SearchRequest.TypeBinding(type));
                } else {
                    searchRequest.getBinding().setType(type);                    
                }
            }
            if (shouldAddResults(searchRequest)) {
                final String query = getQuery(searchRequest);
                final ESearchResult result = dbProvider.query(query, request.getStartIndex(), request.getMaxResults());
                counts.setClassCount(type, result.count);
            }
        }
        return counts;        
    }
    
    public abstract EIEntity getType();
        
    public abstract boolean shouldAddResults(SearchRequest request) throws IOException;
    
    protected abstract String getQuery(SearchRequest request) throws IOException;
    
    protected abstract SearchResult createSearchResultForDocSummary(ESummaryResult sum) throws IOException; 

    /**
     * Gets the type from a SearchRequest.
     * @param request Request
     * @return Type uri
     */
    protected static EIURI getType(SearchRequest request) {
        SearchRequest.TypeBinding binding = request.getBinding();
        if (binding != null) {
            return binding.getType();
        }
        SearchRequest.Term term = request.getTerm();
        if (term != null && term.getURI() != null) {
            return term.getURI();
        }
        return null;
    }
    
    /**
     * Checks, with inference, if a given type is of another type.
     */
    protected boolean isType(EIURI type, EIURI target) {
        if (type == null || target == null) {
            return false;
        }
        if (type.equals(target)) {
            return true;
        }
        final EIClass eiClass = eagleiOntModel.getClass(target);
        if (eiClass != null && eiClass.hasSubClass()) {
            for (EIClass sub: eagleiOntModel.getSubClasses(target)) {
                if (isType(type, sub.getEntity().getURI())) {
                    return true;
                }
            }
        }
        return false;
    }

    /**
     * Generic query generation logic
     */
    protected String getQuery(final SearchRequest request, final boolean includeType, final Set<EIURI> typesToIgnore, final boolean includeInstitution) {
        StringBuilder sb = new StringBuilder();
        SearchRequest.Term term = request.getTerm();
        if (term != null) {
            if (term.getQuery() != null) {
                sb.append(term.getQuery() + " ");
            }
            if (term.getURI() != null) {
                for (String label: eagleiOntModel.getLabels(term.getURI())) {
                    sb.append(label + " ");
                }
            }
            // TODO add term URI labels
        } 
        SearchRequest.TypeBinding binding = request.getBinding();
        if (binding != null && includeType) {
            EIURI uri = binding.getType();
            if (!typesToIgnore.contains(uri)) {
                for (String label: eagleiOntModel.getLabels(uri)) {
                    sb.append(label + " ");                
                }
            }
        }

        if (request.getProviderInstitution() != null && includeInstitution) {
            EIEntity entity = this.institutionRegistry.getInstitution(request.getProviderInstitution());
            sb.append(entity.getLabel());
        }

        if (DEBUG) {
            logger.debug("Using NCBI query: " + sb.toString());
        }
        
        return sb.toString();
    }        
}
