package org.eaglei.search.provider.ncbi;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.w3c.dom.Document;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Contains logic for using NCBI EUtils.
 * 
 * TODO: try to switch to the SOAP-based interface via axis2 or an equivalent framework
 * 
 * @author rfrost
 */
public class EUtils {

    private static final Log logger = LogFactory.getLog(EUtils.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    public static final String EUTILS_URL = "http://eutils.ncbi.nlm.nih.gov/entrez/eutils/";
    public static final String DB = "db=";
    public static final String TERM = "term=";
    public static final String RETMAX = "retmax=";
    public static final String RETSTART = "retstart=";    
    public static final String TOOL = "tool=";    
    public static final String ID_PARAM= "id=";
    public static final String RETMODE= "retmode=";
    public static final String IDLIST= "IdList";
    public static final String ID= "Id";    
    
    /**
     * Executes an NCBI request at the specified eUtils URL and returns the XML response as a DOM Document
     * @param url eUtils URL with all params.
     * @return Response as DOM Document.
     * @throws IOException Thrown if there is an error.
     */
    public static Document executeNCBIRequest(final String url) throws IOException {
        final HttpClient httpclient = new HttpClient();
        final GetMethod httpget = new GetMethod(url);
        InputStream is = null;
        try {
            httpclient.executeMethod(httpget);
            is = httpget.getResponseBodyAsStream();
            Document doc = parse(is);
            //if (DEBUG) {
            //logger.debug(serializeDocument(doc));
            //}
            return doc;
        } finally {
            httpget.releaseConnection();
            if (is != null) {
                is.close();
            }
        }        
    }        
    
    private static Document parse(final InputStream stream) throws IOException {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            return builder.parse(new InputSource(stream));
        } catch (SAXException saxe) {
            throw new IOException(saxe);
        } catch (ParserConfigurationException pce) {
            throw new IOException(pce);
        }
    }
    
    /**
     * Builds a query URL for executing an eUtils service that retrieves information for a set of IDs.
     * @param serviceURL Base URL for the service
     * @param ids IDs of artifacts whose information is being retrieved
     * @param db The target db
     * @param max Max number of results
     * @param tool The source tool
     * @return Query URL.
     */
    public static String buildServiceURL(final String serviceURL, final List<String> ids, final String db, final int max, final String tool) {
        StringBuilder sb = new StringBuilder();
        sb.append(serviceURL);
        sb.append("&" +DB + db);
        sb.append("&" +TOOL + tool);
        sb.append("&" + ID_PARAM);
        for (int i = 0; i < max && i < ids.size(); i++) {
            if (i > 0) {
                sb.append(",");
            }
            sb.append(ids.get(i));
        }
        sb.append("&" +RETMODE + "xml");            
        sb.append("&" +RETMAX + max);
        return sb.toString();
    }
    
    /**
     * Utility method to serialize an XML document.
     * @param doc The DOM document.
     * @return String serialization.
     */
    public static String serializeDocument(final Document doc) {
        try {
            DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();
            DOMImplementationLS impl = (DOMImplementationLS)registry.getDOMImplementation("LS");
            LSSerializer writer = impl.createLSSerializer();
            return writer.writeToString(doc);
        } catch (Exception e) {
        }
        return null;
    }
}
