package org.eaglei.search.provider.nif;

import java.io.IOException;
import java.io.InputStream;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.List;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.w3c.dom.bootstrap.DOMImplementationRegistry;
import org.w3c.dom.ls.DOMImplementationLS;
import org.w3c.dom.ls.LSSerializer;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

/**
 * Contains logic for using NIF ESearch.
 *  
 * @author rfrost
 * @author tbashor
 */
public final class NIFQueryUtil {

    public static final String EUTILS_URL = "http://eutils.ncbi.nlm.nih.gov/entrez/eutils/";
    public static final String DB = "db=";
    public static final String TERM = "term=";
    public static final String RETMAX = "retmax=";
    public static final String RETSTART = "retstart=";    
    public static final String TOOL = "tool=";    
    public static final String ID_PARAM= "id=";
    public static final String RETMODE= "retmode=";
    public static final String IDLIST= "IdList";
    public static final String ID= "Id";    
    
    private static final Log logger = LogFactory.getLog(NIFQueryUtil.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final String EUTILS_SEARCH_URL = EUTILS_URL + "esearch.fcgi?";
    
    private static final String NIF_SEARCH_URL = "http://nif-services.neuinfo.org/nif/services/federationQuery/";
    private static final String NIF_MOUSE_DB = "IMSR/Mice";

    /**
     * Executes an NCBI search.
     */
    public static Document search(final String query, final String db, final String tool, final int startIndex, final int maxResults) throws IOException {
        final String searchURL = buildSearchURL(query, db, tool, startIndex, maxResults); 
        if (DEBUG) {
            logger.debug("Querying NIF search service at " + searchURL);
        }
     
        Document doc = executeNIFRequest(searchURL);
        return doc;
    }
    
    /**
     * 
     */
    public static String buildSearchURL(final String query, final String db, final String tool, final int start, final int max) {
        StringBuilder sb = new StringBuilder();
        sb.append(NIF_SEARCH_URL);
        sb.append(NIF_MOUSE_DB);
        sb.append("?");
        sb.append("q=" + URLEncoder.encode(query));
        sb.append("&");
        sb.append("offset=" + start);
        sb.append("&");
        sb.append("count=" + max);
        return sb.toString();
    }

    /**
     * Simple representation of an eSearch result and, optionally, the results of a subsequent
     * set of sSummary or eFetch calls.
     */
    public static class ESearchResult {
        /**
         * Total number of results returned
         */
        public int count;
        /**
         * Start of the result set
         */
        public int start;
        /**
         * Max number of results
         */
        public int max;
        /**
         * Translation of the query
         */
        public String queryTranslation;
        /**
         * IDs of result set.
         */
        public List<String> ids = new ArrayList<String>();
        /**
         * XML Document that represent the associated eSummary or eFetch results.
         */
        public Document details;
        
        /**
         * Creates an ESearchResult from an XML Document.
         * @param doc
         * @throws IOException
         */
        public ESearchResult(Document doc) throws IOException {
            final Element eSearchResult = doc.getDocumentElement();
            final NodeList children = eSearchResult.getChildNodes();
            for (int i = 0; i < children.getLength(); i++) {
                final Node node = children.item(i);
                final String nodeName = node.getNodeName();
                if (nodeName.equals(IDLIST)) {
                    final NodeList idNodes = node.getChildNodes();
                    for (int j = 0; j < idNodes.getLength(); j++) {
                        final Node idNode = idNodes.item(j);
                        if (idNode.getNodeName().equals(ID)) {
                            this.ids.add(idNode.getTextContent());
                        }
                    }
                } else if(nodeName.equals("Count")) {
                    this.count = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("RetMax")) {            
                    this.max = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("RetStart")) {            
                    this.start = Integer.parseInt(node.getTextContent());
                } else if(nodeName.equals("QueryTranslation")) {            
                    this.queryTranslation = node.getTextContent();
                }
            }
        }
    }
    
    /**
     * Executes an NCBI request at the specified eUtils URL and returns the XML response as a DOM Document
     * @param url eUtils URL with all params.
     * @return Response as DOM Document.
     * @throws IOException Thrown if there is an error.
     */
    public static Document executeNIFRequest(final String url) throws IOException {
        final HttpClient httpclient = new HttpClient();
        httpclient.getParams().setParameter("accept", "application/xml");
        final GetMethod httpget = new GetMethod(url);
        InputStream is = null;
        try {
            httpclient.executeMethod(httpget);
            is = httpget.getResponseBodyAsStream();
            Document doc = parse(is);
            //if (DEBUG) {
            //  logger.debug(serializeDocument(doc));
            //}
            return doc;
        } finally {
            httpget.releaseConnection();
            if (is != null) {
                is.close();
            }
        }        
    }        
    
    private static Document parse(final InputStream stream) throws IOException {
        try {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            return builder.parse(new InputSource(stream));
        } catch (SAXException saxe) {
            throw new IOException(saxe);
        } catch (ParserConfigurationException pce) {
            throw new IOException(pce);
        }
    }
    
    /**
     * Builds a query URL for executing an eUtils service that retrieves information for a set of IDs.
     * @param serviceURL Base URL for the service
     * @param ids IDs of artifacts whose information is being retrieved
     * @param db The target db
     * @param max Max number of results
     * @param tool The source tool
     * @return Query URL.
     */
    public static String buildServiceURL(final String serviceURL, final List<String> ids, final String db, final int max, final String tool) {
        StringBuilder sb = new StringBuilder();
        sb.append(serviceURL);
        sb.append("&" +DB + db);
        sb.append("&" +TOOL + tool);
        sb.append("&" + ID_PARAM);
        for (int i = 0; i < max && i < ids.size(); i++) {
            if (i > 0) {
                sb.append(",");
            }
            sb.append(ids.get(i));
        }
        sb.append("&" +RETMODE + "xml");            
        sb.append("&" +RETMAX + max);
        return sb.toString();
    }
    
    /**
     * Utility method to serialize an XML document.
     * @param doc The DOM document.
     * @return String serialization.
     */
    public static String serializeDocument(final Document doc) {
        try {
            DOMImplementationRegistry registry = DOMImplementationRegistry.newInstance();
            DOMImplementationLS impl = (DOMImplementationLS)registry.getDOMImplementation("LS");
            LSSerializer writer = impl.createLSSerializer();
            return writer.writeToString(doc);
        } catch (Exception e) {
        }
        return null;
    }

}
