package org.eaglei.search.provider.rdf;

import java.io.File;
import java.io.FileWriter;
import java.io.IOException;
import java.io.Writer;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.jena.EagleIOntUtils;
import org.eaglei.search.datagen.DataGenParams;
import org.eaglei.services.InstitutionRegistry;
import org.springframework.context.ApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.hp.hpl.jena.ontology.OntResource;
import com.hp.hpl.jena.rdf.model.Model;

/**
 * Uses the RDFGenerator to create test RDF data and serializes it to the file system. 
 * 
 * @see RDFGenerator
 */
public class RDFFileGenerator {

    private static final Log logger = LogFactory.getLog(RDFFileGenerator.class);
    
    /**
     * Executes RDFGenerator to generate RDF for all institutions to the specified
     * output directory. The RDF for each institution is output in a separate file.
     * @param args args[0] Path where output RDF should be stored.
     * @throws IOException Thrown if an exception is encountered generating the test data
     */
    public static void main(String[] args) throws Exception {
        if (args.length == 0 || args[0].length() == 0) {
            logger.error("RDFFileGenerator <output RDF path>");
            return;
        }
        // create and configure beans
        ApplicationContext context =
            new ClassPathXmlApplicationContext(new String[] {"search-rdf-config.xml"});

        // retrieve configured instance
        InstitutionRegistry institutionRegistry = context.getBean(InstitutionRegistry.class);
        EIOntModel eagleiOntModel = context.getBean(EIOntModel.class);
        generate(args[0], institutionRegistry, eagleiOntModel);
    }
    
    /**
     * Executes RDFGenerator to generate RDF for all institutions to the specified
     * output directory. The RDF for each institution is output in a separate file.
     * @param outputPath Path where output RDF should be stored.
     * @throws IOException Thrown if an exception is encountered generating the test data
     */
    public static void generate(String outputPath, InstitutionRegistry institutionRegistry, EIOntModel eagleiOntModel) throws IOException {
        assert outputPath != null;

        final File outputDir= new File(outputPath);
        
        logger.debug("Generating test RDF data to directory: " + outputDir.getAbsolutePath());
        
        // make certain the dir exists or we can create it
        if (!outputDir.exists()) {
            if (!outputDir.mkdirs()) {
                logger.error("Failed to create output directory " + outputDir.getAbsolutePath());
                return;
            }
        }
        
        // Generate for each institution 
        List<EIEntity> institutions = institutionRegistry.getInstitutions();
        for (EIEntity institution : institutions) {
            final String uri = institution.getURI().toString();
            final String label = institution.getLabel();
            final DataGenParams genParams = DataGenParams.createForOneInstitution(uri, label);
            final RDFGenerator generator = new RDFGenerator(eagleiOntModel);

            // generate the test RDF
            logger.debug("Generating for institution: " + label);
            generator.generate(genParams);
            
            // get the Jena Model that holds the RDF 
            final Model rdf = generator.getModel();
            logger.debug("Generated " + rdf.listStatements().toList().size() + " triples for " + label); 

            // write out the model
            final File rdfFile = new File(outputDir, label + ".rdf");
            logger.debug("Serializing RDF to " + rdfFile.getAbsolutePath() + "...");
            final Writer writer = new FileWriter(rdfFile);
            try {
                rdf.write(writer, "RDF/XML");
                writer.flush();
            } finally {
                writer.close();
            }
            logger.debug("...finished writing RDF to " + rdfFile.getAbsolutePath());
        }
    }    
}
