package org.eaglei.search.provider.rdf;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.provider.SearchResult;

import com.hp.hpl.jena.query.Query;
import com.hp.hpl.jena.query.QueryExecution;
import com.hp.hpl.jena.query.QueryExecutionFactory;
import com.hp.hpl.jena.rdf.model.Model;
import com.hp.hpl.jena.sparql.engine.http.QueryEngineHTTP;

/**
 * AbstractRDFProvider subclass that executes the SPARQL query 
 * against a SPARQL endpoint.
 */
public class SPARQLProtocolProvider extends AbstractRDFProvider {
     
    private static final Log logger = LogFactory.getLog(SPARQLProtocolProvider.class);
    
    /**
      * Creates a SearchProvider for an RDF-repository that is accessed via SPARQL Protocol
      * at a URL that is retrieved via the System property "search.provider.sparql.url".
      * The username and password for basic http authentication are retrieved via the
      * System properties "search.provider.sparql.http.username" and "search.provider.sparql.http.username".
      * If these are not specified, basic authentication is not used.
      * 
      * @throws IOException Thrown if an error is encountered.
      */
    /*
    public SPARQLProtocolProvider() throws IOException {
        this(System.getProperty(SPARQL_URL), System.getProperty(SPARQL_HTTP_USERNAME), System.getProperty(SPARQL_HTTP_PASSWORD));        
    }
    */
    
    /**
     * Creates a SearchProvider for an RDF-repository that is accessed via SPARQL Protocol at the 
     * specified URL.
     * 
     * @param url URL of the SPARQL endpoint.
     * 
     * @throws IOException Thrown if an error is encountered.
     */
    /*
    public SPARQLProtocolProvider(final EIOntModel eagleiOntModel, final EIEntity institution, final String url) throws IOException {
        super(eagleiOntModel, null);
        assert url != null;
        this.url = url;
        this.setDefaultGraphURI(PUBLISHED_NG);
    }
    */
    
    /**
     * Creates a SearchProvider for an RDF-repository that is accessed via SPARQL Protocol at the 
     * specified URL.
     * 
     */
    public SPARQLProtocolProvider(final EIOntModel eagleiOntModel, final EIEntity institution,
            final RepositoryConfig repoConfig) {
        super(eagleiOntModel, institution, repoConfig);
    }
    
    @Override
    protected SearchResult createSearchResultFromModel(final String uri, final Model model) {    
        SearchResult result = super.createSearchResultFromModel(uri, model);
    	if (result != null) {
    	    // set URL to URI for result entity
    	    result.setURL(result.getEntity().toString());
    	}
    	return result;
    }
    
    @Override
    protected QueryExecution getQueryExecution(final Query query) {
        // cast to QueryEngineHTTP to allow us to set the basic http auth credentials and default graph URI
        final QueryEngineHTTP qe = (QueryEngineHTTP) QueryExecutionFactory.sparqlService(repoConfig.getUrl(), query);
        if (repoConfig.getUsername() != null && repoConfig.getPassword() != null) {
        	qe.setBasicAuthentication(repoConfig.getUsername(), repoConfig.getPassword().toCharArray());
            if (repoConfig.getDefaultGraphURI() != null) {
                qe.addDefaultGraph(repoConfig.getDefaultGraphURI());
            }
        }
        
        return qe;
    }
}
