package org.eaglei.services.repository;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringReader;
import java.io.StringWriter;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.UUID;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.httpclient.methods.GetMethod;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIURI;
import org.eaglei.security.SecurityProvider;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;
import com.hp.hpl.jena.rdf.model.Literal;
import com.hp.hpl.jena.rdf.model.Resource;

/**
 * 
 * @author Ricardo De Lima
 * @author Lucy Hadden
 * @author Daniela Bourges
 * 
 *         April 11, 2010
 * 
 *         Center for Biomedical Informatics (CBMI)
 * @link https://cbmi.med.harvard.edu/
 * 
 * 
 * 
 */
public final class RepositorySecurityProvider implements SecurityProvider {

    private static final Log log = LogFactory.getLog(RepositorySecurityProvider.class);
    private static final boolean isDebugEnabled = log.isDebugEnabled();

    private final Map<String, RepositoryHttpClient> mapInstIdToRepoClient = new HashMap<String, RepositoryHttpClient>();
    private Map<String, Session> mapIdToSession = new HashMap<String, Session>();
    private Map<String, HttpClient> mapIdToHttpClient = new HashMap<String, HttpClient>();

    public RepositorySecurityProvider() {
        // This constructor is provided for the convenience of datatools 
        // which might not be fully spring-ified yet, and thus might
        // not have a handle to the InstitutionRegistry bean.
        // Only valid for institutional node deployment.
        RepositoryHttpClient config = new RepositoryHttpClient(System.getProperty("org.eaglei.subdomain"));
        mapInstIdToRepoClient.put(null, config);
    }
    
    /**
     * Intitialize from InstitutionRegistry configuration info.
     */
    public RepositorySecurityProvider(InstitutionRegistry institutionRegistry) {
        if (institutionRegistry.getInstitution() != null) {
            // Institutional node
            RepositoryHttpClient config = new RepositoryHttpClient(institutionRegistry.getSubdomain());
            mapInstIdToRepoClient.put(null, config);
            // Probably won't get used, but add lookup from uri
            String uriStr = institutionRegistry.getMapSubdomainToURI().get(institutionRegistry);
            mapInstIdToRepoClient.put(uriStr, config);
        } else {
            // Central search node
            for (Map.Entry<String,String> entry : institutionRegistry.getMapSubdomainToURI().entrySet()) {
                // For now assuming same superuser user/pass for all repos...
                // Obviously, this will need to change when there are individual repo user/pass
                RepositoryHttpClient config = new RepositoryHttpClient(entry.getKey());
                mapInstIdToRepoClient.put(entry.getValue(), config);
            }
        }
    }

    @Override
    public Session logIn(final String institutionId, final String user, final String password) {
        HttpClient client = getHttpClient(user, password);
        // null institution id param is OK, on an institutional node
        RepositoryHttpClient repoClient = mapInstIdToRepoClient.get(institutionId);
        if (repoClient == null) {
            log.error("Unrecognized login institution id: "+institutionId);
            return null;
        }

        int status = 0;
        ResultSet results = null;
        String responseBody = null;
        final GetMethod method = new GetMethod(repoClient.getWhoamiUrl());
        try {
            if (isDebugEnabled) {
                log.debug("Authenticating user " + user + " at " + repoClient.getWhoamiUrl());
            }

            status = client.executeMethod(method);
            responseBody = getStringFromInputStream(method.getResponseBodyAsStream());
            if (status != HttpStatus.SC_OK) {
                if (status == HttpStatus.SC_NOT_FOUND) {
                    log.error("Repo unavailable");
                } else if (status == HttpStatus.SC_UNAUTHORIZED) {
                    // This is the only expected "failure".  Should it be handled differently?
                    log.info("not authorized to get user information (login/whoami) : " + user);
                } else {
                    log.error(responseBody);
                }
                return null;
            }
            
            results = ResultSetFactory.fromXML(responseBody);
            if (results == null) {
                log.error("whoami response body is null");
                return null;
            }
            
            // we are authorized so let's return the user name
            final QuerySolution soln = results.nextSolution();
            final Literal username = soln.getLiteral("username");
            if (username == null) {
                log.error("Could not authenticate, null username in response body: " + user);
                return null;
            }
            final Resource userURI = soln.getResource("uri");
            if (userURI == null) {
                log.error("Could not authenticate, null uri in response body: " + user);
                return null;
            }

            // Repo right now doesn't contain First/Last names so let's just
            // return the username and URI

            if (isDebugEnabled) {
                log.debug("Authenticated user: " + username.getString() + " " + userURI.getURI());
            }

            // I read somewhere that this was a good way to generate session ids.
            //SecureRandom random = new SecureRandom();
            //byte bytes[] = new byte[20];
            //random.nextBytes(bytes);
            //String sessionId = String.valueOf(bytes);
            String sessionId = UUID.randomUUID().toString();
            //List<String> states = getWFStatesList(sessionId, userURI.getURI());
            Session session = new Session(sessionId, institutionId, username.getString(), userURI.getURI());
            mapIdToSession.put(sessionId, session);
            mapIdToHttpClient.put(sessionId, client);
            return session;           
        } catch (final Exception e) {
            log.error("problem getting user info " + repoClient.getWhoamiUrl() + " Message from repo: "
                    + responseBody + "; Exception "
                    + e);
            return null;
        } finally {
            method.releaseConnection();
        }
    }

    @Override
    public void logOut(String sessionId) {
        PostMethod method = null;
        try {
            HttpClient client = mapIdToHttpClient.get(sessionId);
            Session session = mapIdToSession.get(sessionId);
            if (client == null) {
                return;
            }
            if (session == null) {
                return;
            }
            RepositoryHttpClient repoClient = mapInstIdToRepoClient.get(session.getInstitutionId());
            method = new PostMethod(repoClient.getLogoutUrl());
            int status = 0;
            if (isDebugEnabled) {
                log.debug("Trying to logout at " + repoClient.getLogoutUrl());
            }
            status = client.executeMethod(method);
            // check to see if we succeeded
            if (status == HttpStatus.SC_OK) {
                if (isDebugEnabled) {
                    log.debug("logout succeded");
                }
            } else {
                log.info("Could not logout user: HTTP Status: " + status);
            }
        } catch (final Exception e) {
            log.error("Unexpected error during logout  " + e);
        } finally {
            if (method != null) {
                method.releaseConnection();
            }
            mapIdToHttpClient.remove(sessionId);
            mapIdToSession.remove(sessionId);
        }
    }

    private HttpClient getHttpClient(String username, String password) {
        HttpClient client = new HttpClient();
        client.setHttpConnectionManager(new MultiThreadedHttpConnectionManager());
        client.getState().setCredentials(AuthScope.ANY,
                new UsernamePasswordCredentials(username, password));
        client.getParams().setParameter("accept", "application/sparql-results+xml");
        client.getParams().setAuthenticationPreemptive(true);
        return client;
    }
    
    private String getStringFromInputStream(InputStream in) throws IOException {
        StringWriter writer = new StringWriter();
        //encoding needs to be explicitly set
        IOUtils.copy(in, writer, "UTF-8");
        return writer.toString();
    }

    @Override
    public Session getSession(String sessionId) {
        return mapIdToSession.get(sessionId);
    }
}
