package org.eaglei.services;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.TreeMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.services.repository.RepositoryHttpConfig;

public class InstitutionRegistry {
    
    public static final String EAGLEI_TIER_PROPERTY = "org.eaglei.tier";
    public static final String EAGLEI_SUBDOMAIN_PROPERTY = "org.eaglei.subdomain";

    private static final Log log = LogFactory.getLog(InstitutionRegistry.class);
    
    public static final String CENTRAL_SEARCH_NODEID = "search";
    public static final String PRODUCTION_TIER = "prod";
    
    private String tier;
    private String localNodeId;
    List<NodeConfig> listNodeConfig;
    private Map<String, NodeConfig> mapNodeIdToConfig = new HashMap<String, NodeConfig>();
    private Map<String, NodeConfig> mapURIToConfig = new HashMap<String, NodeConfig>();
    private final Map<String, RepositoryHttpConfig> mapUriToRepoConfig = new HashMap<String, RepositoryHttpConfig>();
    private List<EIEntity> institutions;
    
    public InstitutionRegistry() {
    }
    
    public void setTier(String tier) {
        this.tier = tier;
    }
    
    public String getTier() {
        return this.tier;
    }
    
    public void setLocalNodeId(String localNodeId) {
        this.localNodeId = localNodeId;
    }
    
    public String getLocalNodeId() {
        return this.localNodeId;
    }
    
    public void setNodeConfigs(List<NodeConfig> nodeConfigs) {
        this.listNodeConfig = nodeConfigs;
        // Generate an alpha sorted institutions list
        TreeMap<String, EIEntity> mapLabelToEntity = new TreeMap<String, EIEntity>();
        for (NodeConfig config : listNodeConfig) {
            EIEntity entity = config.getEntity();
            mapLabelToEntity.put(config.getLabel(), entity);
            mapNodeIdToConfig.put(config.getNodeId(), config);
            mapURIToConfig.put(config.getUri(), config);
        }
        institutions = new ArrayList<EIEntity>(mapLabelToEntity.values());
    }

    public List<NodeConfig> getNodeConfigs() {
        return listNodeConfig;
    }
    
    /**
     * Lookup an institution entity by uri.
     * 
     * @param uri
     * @return
     */
    public EIEntity getInstitution(EIURI uri) {
        NodeConfig config = mapURIToConfig.get(uri.toString());
        return (config != null ? config.getEntity() : null);
    }
    
    /**
     * Get the institution entity of the application's node.
     * Null will be returned if running as the central search node.
     * 
     * @return
     */
    public EIEntity getInstitution() {
        if (CENTRAL_SEARCH_NODEID.equals(getLocalNodeId())) {
            return null;
        } else {
            return mapNodeIdToConfig.get(getLocalNodeId()).getEntity();
        }
    }

    public List<EIEntity> getInstitutions() {
        return this.institutions;
    }
    
    /**
     * Gets a RepositoryHttpConfig for this institutional node.
     * Error will be thrown if this is a central search node.
     * 
     * @return RepositoryHttpConfig
     */
    public RepositoryHttpConfig getRepositoryHttpConfig() {
        return getRepositoryHttpConfig(null);
    }
    
    public RepositoryHttpConfig getRepositoryHttpConfig(String institutionUri) {
        RepositoryHttpConfig repoConfig = mapUriToRepoConfig.get(institutionUri);
        if (repoConfig == null) {
            NodeConfig nodeConfig;
            if (institutionUri == null) {
                assert(getLocalNodeId() != null);
                nodeConfig = mapNodeIdToConfig.get(getLocalNodeId());
            } else {
                nodeConfig = mapURIToConfig.get(institutionUri);
            }
            // Assume for now that all nodes have exactly one repo
            repoConfig = new RepositoryHttpConfig(nodeConfig, 
                    (getTier() != null && getTier().equals(PRODUCTION_TIER) ? null : getTier()));
            mapUriToRepoConfig.put(institutionUri, repoConfig);
            // If it's the local node repoConfig, ensure that there's an entry
            // for both the uri and null.
            if (institutionUri == null) {
                mapUriToRepoConfig.put(nodeConfig.getUri(), repoConfig);
            } else if (institutionUri.equals(getLocalNodeId())) {
                mapUriToRepoConfig.put(null, repoConfig);
            }
        }
        return repoConfig;
    }
}
