/**
 * The eagle-i consortium
 * Harvard University
 * Nov 4, 2010
 */
package org.eaglei.services.repository;

import static org.eaglei.model.jena.SPARQLConstants.LABEL_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.SUBJECT_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.CREATION_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.IS_STUB_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.MODIFIED_DATE_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.OWNER_NAME_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.OWNER_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_VARIABLE;
import static org.eaglei.model.jena.SPARQLConstants.PROVIDER_NAME_VARIABLE;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.MultiThreadedHttpConnectionManager;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIInstance;
import org.eaglei.model.EIInstanceFactory;
import org.eaglei.model.EIInstanceMinimal;
import org.eaglei.model.EIInstanceX;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIProperty;
import org.eaglei.model.EIURI;
import org.eaglei.model.jena.EIInstanceMinimalFactory;
import org.eaglei.model.jena.SPARQLQueryUtil;
import org.eaglei.search.provider.AuthSearchRequest;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.NodeConfig;

import com.hp.hpl.jena.query.QuerySolution;
import com.hp.hpl.jena.query.ResultSet;
import com.hp.hpl.jena.query.ResultSetFactory;


public class RepositoryInstanceProvider {
	private static final Log log = LogFactory.getLog( RepositoryInstanceProvider.class );
	private static final boolean isDebugEnabled = log.isDebugEnabled();
	//TODO these used to be in ProviderMessages - do we pull that out too?
	private static final String NO_SESSION_MESSAGE = "No session information was found.";
	private static final String FAILURE_MESSAGE = "Could not complete operation; repository message is: ";
	
    protected static final String READ_VIEW = "user";
	// string sent as part of http requests
	private static final String FORMAT_VALUE = "application/xml";
	// format string used for received RDF
	private static final String RDF_FORMAT = "RDF/XML";
	
	private EIOntModel eiOntModel;
	private RepositorySecurityProvider securityProvider;
	private EIInstanceFactory instanceFactory;
    // Use connection pooling, will try to keep connections open between uses.
    private MultiThreadedHttpConnectionManager connectionManager = new MultiThreadedHttpConnectionManager();
    private Map<String, RepositoryHttpConfig> mapHostToRepoConfig = new HashMap<String, RepositoryHttpConfig>();
	private Map<String, HttpClient> mapHostToHttpClient = new HashMap<String, HttpClient>();
    /*
     * HttpClient that is reused. It will reuse the underlying connection or recreate if it gets dropped.
     */
    private HttpClient httpclient;
    
    protected enum RestCommands {
		// IMPORTANT logout needs a / at the end
		GetNewInstanceIDs("repository/new"), WhoAmI("repository/whoami"), Logout("repository/logout/"), GetInstance("i"), UpdateInstance("repository/update"), Query("repository/sparql"), Online(""), Graph("repository/graph"), FakeWorkflow(
				"repository/fakeworkflow"), Listgraphs("repository/listGraphs"), Claim("repository/workflow/claim"), Release("repository/workflow/release"), Transition("repository/workflow/push"), ListTransitions("repository/workflow/transitions"), ListResources(
				"repository/workflow/resources");

		private RestCommands(final String propKey) {
			key = propKey;
		}

		public String getURL() {
			return key;//DEFAULT_REPOSITORY + key;
		}

		private final String key;
	}
	
	public RepositoryInstanceProvider(EIOntModel eiOntModel, RepositorySecurityProvider provider, InstitutionRegistry registry, EIInstanceFactory instanceFactory) {
	    this.eiOntModel = eiOntModel;
		this.securityProvider = provider;
		this.instanceFactory = instanceFactory;
		// Build a lookup table of instance id prefix to search user HttpClient
		for (NodeConfig nodeConfig : registry.getNodeConfigs()) {
		    RepositoryHttpConfig repoConfig = registry.getRepositoryHttpConfig(nodeConfig.getUri());
		    String hostURL = repoConfig.getHostURL();
		    hostURL = hostURL.replace("https", "http");	        
	        httpclient = RepositoryHttpConfig.createHttpClient(repoConfig.getSearchUsername(), repoConfig.getSearchPassword());
	        httpclient.setHttpConnectionManager(connectionManager);
	        nodeConfig.getRepositoryHostUrl();
            mapHostToRepoConfig.put(hostURL, repoConfig);
	        mapHostToHttpClient.put(hostURL, httpclient);
		}
	}
	
	
	//TODO Eventually we need to get rid of this method as it refers to EIInstanceX. Keeping it for now as this is being used for tooltips. 
	public EIInstanceX getInstance(final String sessionID, final EIURI instanceID) throws Exception {
        if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }

		if ( instanceID == null || instanceID.toString() == null ) {
			log.warn( "Trying to get instance for null EIURI" );
			return null;
		}
		
		String instanceIDStr = instanceID.toString();
		RepositoryHttpConfig repoConfig = getRepoConfigFromInstanceId( instanceIDStr );
        if (repoConfig == null) {
		    log.error("Unrecognized resource instance URL:  " + instanceIDStr);
		    throw new Exception("Unrecognized resource instance URL:  " + instanceIDStr);
		}
        
        String host = instanceIDStr.substring(0, instanceIDStr.indexOf("/i/") + 1);
        HttpClient searchHttpClient = mapHostToHttpClient.get(host);
		
		int status = 0;
		final PostMethod method = new PostMethod( repoConfig.getInstanceUrl() );
		setReadParameters( method );
		method.setParameter( "uri", instanceIDStr );
		if ( isDebugEnabled ) {
			log.debug( "Trying to get instance " + instanceID.toString() );
		}
		try {
			status = searchHttpClient.executeMethod( method );
			if ( status == HttpStatus.SC_OK ) {
				final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
				final EIInstance instance = instanceFactory.create( instanceID, response, RDF_FORMAT );
				if ( isDebugEnabled ) {
					log.debug( "got instance contents: " + instance );
				}
				// Get references and add inferred inverses
				//List<EIInstanceMinimal> references = getReferencedBy(searchHttpClient, repoConfig, instanceID, instance);
				// Identify term values
				Map<EIURI, String> mapTermToDefinition = getTerms(instance);
				return new EIInstanceX(instance,  mapTermToDefinition);
			} else {
				log.error( "get instance " + instanceID + " failed with status: " + status );
				throw new RepositoryProviderException(FAILURE_MESSAGE + "get instance, " +status );
			}
		} finally {
			method.releaseConnection();
			// Take the opportunity to close any connections that haven't been used in half an hour.
			// TODO should have reaper thread that does this.
			connectionManager.closeIdleConnections(1800000);
		}
	}

	public RepositoryHttpConfig getRepoConfigFromInstanceId(String instanceIDStr) {
		String host = instanceIDStr.substring(0, instanceIDStr.indexOf("/i/") + 1);
        RepositoryHttpConfig repoConfig = mapHostToRepoConfig.get(host);
        
        return repoConfig;
	}

	public EIInstance getEIInstance(final String sessionID, final EIURI instanceUri) throws RepositoryProviderException {
		if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }
		
		if ( instanceUri == null || instanceUri.toString() == null || EIURI.NULL_EIURI.equals( instanceUri ) ) {
			log.warn( "Trying to get instance for null EIURI" );
			throw new RepositoryProviderException( "Trying to get instance for null EIURI" );
		}
		final PostMethod method = new PostMethod( getRepoConfigFromInstanceId(instanceUri.toString()).getHostURL() + RestCommands.GetInstance.getURL());
		setReadParameters( method );
		method.setParameter( "uri", instanceUri.toString() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to get instance " + instanceUri.toString() );
		}
		final String responseBody = ProviderUtils.getHttpResponse(httpclient, method );
		EIInstance eiInstance;
		eiInstance = instanceFactory.createExtended( instanceUri, responseBody, RDF_FORMAT );
		
		
		if ( isNull( eiInstance ) ) {
			throw new RepositoryProviderException( "Unable to create EIInstance for instanceUri: " + instanceUri );
		}
		eiInstance.setStubs( getAllStubEntities( sessionID, eiInstance ) );
		return eiInstance;
	}
	
	protected Set<EIEntity> getAllStubEntities(final String sessionID, final EIInstance instance) throws RepositoryProviderException {
		final Set<EIEntity> stubs = getStubEntities( sessionID, instance.getInstanceURI() );
		if ( instance.hasEmbeddedInstances() ) {
			for (EIURI embeddedUri : instance.getEmbeddedInstanceUriList()) {
				stubs.addAll( getStubEntities( sessionID, embeddedUri ) );
			}
		}

		return stubs;
	}
	
	//TODO Copied from RESTRepositoryProvider but w/o SortByProperties
	public List<EIInstanceMinimal> listReferencingResources(final String sessionID, final String userURI, final EIURI resourceUri, final AuthSearchRequest queryRequest, final boolean strictOwnerFilter) throws RepositoryProviderException {
		if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }
		final PostMethod method = new PostMethod( getRepoConfigFromInstanceId(resourceUri.toString()).getHostURL() + RestCommands.ListResources.getURL() );
		method.setParameter( "state", "all" );
		final StringBuilder sparql = new StringBuilder();
		// Restrict to eagle-i resources
		sparql.append( SPARQLQueryUtil.getInstance().allTypesPattern( false ) );
		// Include lab info
		sparql.append( SPARQLQueryUtil.getInstance().providerRestrictionPattern( EIURI.NULL_EIURI ) );
		// Restrict to referenced resources
		sparql.append( SPARQLQueryUtil.getInstance().referencedByPattern( resourceUri ) );

		commonListResourcesMethodSetup( EIURI.create( userURI ), queryRequest, "", strictOwnerFilter, sparql, method );

		final ResultSet resultSet = ResultSetFactory.fromXML( ProviderUtils.getHttpResponse( httpclient, method ) );
		final List<EIInstanceMinimal> results = EIInstanceMinimalFactory.getInstance().create( resultSet );
		return results;
	}
	
	
	//TODO Copied from RESTRepositoryProvider. Delete this method from there
	private void commonListResourcesMethodSetup(final EIURI user, final AuthSearchRequest queryRequest, final String sortOrder, final boolean strictOwnerFilter, final StringBuilder sparql, final PostMethod method) {
		method.setParameter( "detail", "full" );
		method.setParameter( "format", FORMAT_VALUE );

		// TODO see at the UI level if other combinations are needed (e.g. owner=self + unclaimed=true for all resources available to user)
		if ( strictOwnerFilter ) {
			method.setParameter( "owner", "self" );
			method.setParameter( "unclaimed", "false" );
		} else {
			method.setParameter( "owner", "all" );
			method.setParameter( "unclaimed", "true" );
		}

		sparql.append( SPARQLQueryUtil.getInstance().modifiedDatePattern() );

		// FIXME this is a fix for the erroneously inserted dcterms:created triples
		// should be removed once the data is cleaned up
		sparql.append( " filter(!regex(str(?" ).append( CREATION_DATE_VARIABLE ).append( "), \"http://www.w3.org/2001/XMLSchema#dateTime\"))" );
		// FIXME fix for resources with no labels (needs cleanup)
		sparql.append( " filter(bound(?" ).append( LABEL_VARIABLE ).append( "))" );

		if ( isDebugEnabled ) {
			log.debug( "Sparql pattern sent to workflow/resources is: " + sparql.toString() );

		}

		try {
			method.setParameter( "addPattern", encodeToUTF8( sparql.toString() ) );
		} catch (UnsupportedEncodingException e) {
			log.warn( "could not encode to utf-8: " + e.getMessage() );
			log.warn( "will send unencoded query string pattern: " + sparql.toString() );
			method.setParameter( "addPattern", sparql.toString() );
		}
		method.setParameter( "addResults", "?" + PROVIDER_VARIABLE + " ?" + PROVIDER_NAME_VARIABLE + " ?" + MODIFIED_DATE_VARIABLE + " ?" + IS_STUB_VARIABLE );
		final StringBuilder modifiers = new StringBuilder();

		// FIXME verify that there are no fence post errors
//		if ( queryRequest.isPaginated() && queryRequest.getMaxResults() > 0 && queryRequest.getStartIndex() >= 0 ) {
//			modifiers.append( "LIMIT " ).append( String.valueOf( queryRequest.getMaxResults() ) ).append( " OFFSET " ).append( String.valueOf( queryRequest.getStartIndex() ) );
//		}
		method.setParameter( "addModifiers", modifiers.toString() );

		if ( isDebugEnabled ) {
			log.debug( "modifiers sent to workflow/resources are: " + modifiers.toString() );

		}
	}
	
	
	private Set<EIEntity> getStubEntities(final String sessionID, final EIURI instanceUri) throws RepositoryProviderException {
		final ResultSet results = ResultSetFactory.fromXML( postQuery( sessionID, SPARQLQueryUtil.getInstance().getRetrieveStubsQuery( instanceUri ), instanceUri.toString() ) );
		final Set<EIEntity> stubs = new HashSet<EIEntity>();
		while ( results.hasNext() ) {
			final QuerySolution solution = results.nextSolution();
			if ( solution.contains( SUBJECT_VARIABLE ) && solution.contains( LABEL_VARIABLE ) ) {
				final EIURI uri = EIURI.create( solution.getResource( SUBJECT_VARIABLE ).getURI() );
				final String label = solution.getLiteral( LABEL_VARIABLE ).getString();
				stubs.add( EIEntity.create( uri, label ) );
			} else {
				continue;
			}
		}
		return stubs;
	}
	

	private boolean isNull(final EIInstance instance) {
		return instance == null || instance.getInstanceURI() == null || EIInstance.NULL_INSTANCE.equals( instance );
	}
	
	public EIInstance setReferencingResources(String sessionId, EIInstance instance) throws RepositoryProviderException {
		final ResultSet results = ResultSetFactory.fromXML( postQuery( sessionId, SPARQLQueryUtil.getInstance().getReferencedByQuery( instance.getInstanceURI() ), instance.getInstanceURI().toString() ) );
		EIInstanceMinimalFactory.getInstance().setReferencingInstances( instance, results );
		return instance;
	}
    
	private synchronized String postQuery(final String sessionId, final String q, final String instanceID) throws RepositoryProviderException {
		return postQuery( sessionId, q, READ_VIEW, instanceID );
	}

	private synchronized String postQuery(final String sessionId, final String q, final String view, final String instanceID) throws RepositoryProviderException {
		securityProvider.isValid( sessionId );

		if ( q == null ) {
			log.warn( "Null query!" );
			return null;
		}
		final PostMethod method = new PostMethod( getRepoConfigFromInstanceId(instanceID).getHostURL()+RestCommands.Query.getURL() );
		if ( isDebugEnabled ) {
			log.debug( "Trying to query: " + getRepoConfigFromInstanceId(instanceID).getHostURL() + RestCommands.Query.getURL());
		}
		setReadParameters( method );
		method.setParameter( "view", view );
		try {
			method.setParameter( "query", encodeToUTF8( q ) );
		} catch (final Exception e) {
			log.warn( "could not encode to utf-8: " + e.getMessage() );
			log.warn( "will send unencoded query string: " + q );
			method.setParameter( "query", q );
		}
		return ProviderUtils.getHttpResponse( httpclient, method );
	}

	private String encodeToUTF8(final String rdfString) throws UnsupportedEncodingException {
		return new String( rdfString.getBytes( "UTF-8" ) );
	}
	
    private Map<EIURI, String> getTerms(final EIInstance instance) {
    	Map<EIURI, String> mapTermToDefinition = null;
    	EIClass eiClass = instance.getInstanceClass();
    	List<EIProperty> properties = eiOntModel.getProperties(eiClass.getEntity().getURI());
    	for (Map.Entry<EIEntity, Set<EIEntity>> entry : instance.getObjectProperties().entrySet()) {
    		List<EIEntity> listTermRanges = getTermRanges(entry.getKey(), properties);
    		if (listTermRanges != null) {
    			// At least one range of this property is a term.
    			// Find  out if any of the values are subtypes.
    			for (EIEntity value : entry.getValue()) {
	    			if (isTermType(value, listTermRanges)) {
	    				if (mapTermToDefinition == null) {
	    					mapTermToDefinition = new HashMap<EIURI, String>();
	    				}
	    				mapTermToDefinition.put(value.getURI(), eiOntModel.getClassDefinition(value.getURI()));
	    			}
    			}
    		}
    	}
    	return mapTermToDefinition;
    }
    
    private List<EIEntity> getTermRanges(EIEntity propEntity, List<EIProperty> properties) {
    	for (EIProperty p : properties) {
    		if (p.getEntity().equals(propEntity)) {
    			EIObjectProperty objProperty = (EIObjectProperty) p;
    			List<EIEntity> termRanges = new ArrayList<EIEntity>(objProperty.getRangeList().size());
    			for (EIClass rangeClass : objProperty.getRangeList()) {
    				if (!rangeClass.isEagleIResource()) {
    					termRanges.add(rangeClass.getEntity());
    				}
    			}
    			return termRanges;
    		}
    	}
    	return null;
    }
    
    private boolean isTermType(EIEntity value, List<EIEntity> listTermRanges) {
    	for (EIEntity range : listTermRanges) {
    		if (range.equals(value)) {
    			return true;
    		} else if (eiOntModel.isSubClass(range.getURI(), value.getURI())) {
    			return true;
    		}
    	}
    	return false;
    }
	
	private void setReadParameters(final PostMethod method) {
		method.setParameter( "format", FORMAT_VALUE );
		method.setParameter( "view", READ_VIEW );
		method.setParameter( "noinferred", "true" );
		method.setRequestHeader( "charset", "UTF-8" );
	}
	
    public boolean contactMessage(final String sessionID, final String client_ip, final EIURI instanceID, String label, boolean test_mode, 
            String from_name, String from_email, String subject, String message) throws Exception {
        if ( !securityProvider.isValid( sessionID ) ) {
            log.error( "Invalid Session - request cannot be completed" );
            throw new RepositoryProviderException( NO_SESSION_MESSAGE );
        }

        if ( instanceID == null || instanceID.toString() == null ) {
            log.warn( "Trying to get instance for null EIURI" );
            return false;
        }
        
        String instanceIDStr = instanceID.toString();
        String host = instanceIDStr.substring(0, instanceIDStr.indexOf("/i/") + 1);
        RepositoryHttpConfig repoConfig = mapHostToRepoConfig.get(host);
        if (repoConfig == null) {
            log.error("Unrecognized resource instance URL:  " + instanceIDStr);
            throw new Exception("Unrecognized resource instance URL:  " + instanceIDStr);
        }
        HttpClient searchHttpClient = mapHostToHttpClient.get(host);
        
        int status = 0;
        final PostMethod method = new PostMethod( repoConfig.getContactUrl() );
        method.setRequestHeader( "charset", "UTF-8" );
        method.setRequestHeader("format", "application/x-www-form-urlencoded" );
        method.setRequestHeader("Referer", repoConfig.getContactUrl().replaceFirst("emailContact", "contact") +
                "?uri="+URLEncoder.encode(instanceIDStr) + 
                "&safe=" + Boolean.toString(test_mode) + 
                "&label=" + URLEncoder.encode(label));
        method.setParameter( "uri", instanceIDStr );
        method.setParameter( "label", label );
        method.setParameter( "test_mode", Boolean.toString(test_mode) );
        method.setParameter( "client_ip", client_ip );
        method.setParameter( "from_name", from_name );
        method.setParameter( "from_email", from_email );
        method.setParameter( "subject", subject );
        method.setParameter( "message", message );
        if ( isDebugEnabled ) {
            log.debug( "Sending contact message " + instanceID.toString() );
        }
        try {
            status = searchHttpClient.executeMethod( method );
            if ( status == HttpStatus.SC_OK ||  status == HttpStatus.SC_MOVED_TEMPORARILY ) {
                return true;
            } else {
                final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
                log.error( "send contact " + instanceID + " failed with status: " + status + "\r\n" + response);
                return false;
            }
        } finally {
            method.releaseConnection();
            // Take the opportunity to close any connections that haven't been used in half an hour.
            // TODO should have reaper thread that does this.
            connectionManager.closeIdleConnections(1800000);
        }
        
    }
    
    
}
