package org.eaglei.services.repository;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.UsernamePasswordCredentials;
import org.apache.commons.httpclient.auth.AuthScope;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.services.NodeConfig;

/**
 * Configuration for connecting to the repository.
 */
public final class RepositoryHttpConfig {

    public static final String EAGLEI_REPOSITORY_USERNAME_PROPERTY = "org.eaglei.repository.username";
    public static final String EAGLEI_REPOSITORY_PASSWORD_PROPERTY = "org.eaglei.repository.password";
    public static final String PUBLISHED_NG = "http://eagle-i.org/ont/repo/1.0/NG_Published";
    public static final String DEFAULT_NG = "http://eagle-i.org/ont/repo/1.0/NG_DefaultWorkspace";
    public static final String PUBLISHED_VIEW = "published-resources";
    private static final Log log = LogFactory.getLog(RepositoryHttpConfig.class);

    private final String hostURL;
    private final String baseURL;
    private String searchUsername;
    private String searchPassword;

    /**
     * Constructs a RepositoryHttpConfig from the given
     * node configuration.
     * 
     * @param tier Optional tier component to be added to the repo URL
     */
    public RepositoryHttpConfig(NodeConfig nodeConfig, String tier) {
        this(computeHostURL(tier, nodeConfig.getRepositoryHostUrl()));
        setSearchUsername(nodeConfig.getSearchUsername());
        setSearchPassword(nodeConfig.getSearchPassword());
    }

    /**
     * Constructs a client using the given host URL.
     * The host URL must not be null and must end with a trailing /.
     * For example, "https://qa.harvard.eagle-i.net/"
     * 
     * @param hostURL may not be null
     */
    public RepositoryHttpConfig(String hostURL) {
    	if (hostURL == null) {
    		throw new RuntimeException("URL may not be null.");
    	}
    	this.hostURL = hostURL;
        baseURL = hostURL + "repository/";
    }
    
    public String getHostURL() {
    	return this.hostURL;
    }
    
    public String getBaseURL() {
    	return baseURL;
    }

    /**
     * Convenience utility method for constructing a default http client
     * suitable for connecting to the repository.
     * 
     * @param username
     * @param password
     * @return
     */
    public static HttpClient createHttpClient(String username, String password) {
        HttpClient client = new HttpClient();
        client.getState().setCredentials(AuthScope.ANY,
                new UsernamePasswordCredentials(username, password));
        client.getParams().setParameter("accept", "application/sparql-results+xml");
        client.getParams().setAuthenticationPreemptive(true);
        return client;
    }

    /**
     * Generates a repository host url using a tier and institution subdomain.
     * Supports default behavior if either is null.
     * 
     * @param tier
     * @param institutionSubdomain
     * @return
     */
    public static String computeHostURL(String tier, String repositoryHostUrl) {
        if (repositoryHostUrl == null) {
        	return ("localhost/");
        } else {
	        StringBuilder buf = new StringBuilder();
	        String domainOnly;
	        if (repositoryHostUrl.startsWith("https://")) {
	        	domainOnly = repositoryHostUrl.substring("https://".length());
	        } else if (repositoryHostUrl.startsWith("http://")) {
                domainOnly = repositoryHostUrl.substring("http://".length());
	        } else {
	            domainOnly = repositoryHostUrl;
	        }
	        buf.append("https://");
            if (tier != null) {
                // Inject tier between // and remainder 
                buf.append(tier.toLowerCase());
                buf.append('.');
            }
            buf.append(domainOnly);
            return buf.toString();
        }
    }
    
    public String getFullRepositoryUrl(RepositoryLocale locale) {
    	if (locale != RepositoryLocale.SEARCH_INSTANCE_URL) {
    		return baseURL + locale.getURL();
    	}
    	else {
    		//Special Case
    		return hostURL + locale.getURL();
    	}
    }

    public String getSearchUsername() {
        return searchUsername;
    }

    public void setSearchUsername(String searchUsername) {
        this.searchUsername = searchUsername;
    }

    public String getSearchPassword() {
        return searchPassword;
    }

    public void setSearchPassword(String searchPassword) {
        this.searchPassword = searchPassword;
    }
    
    public static enum RepositoryLocale{
    	//Important: logout must have / at the end to prevent redirect errors
    	WHOAMI_URL("whoami"), LOGOUT_URL("logout/"), SPARQL_URL("sparql"), HARVEST_URL("harvest"), SEARCH_INSTANCE_URL("i"),
    	CONTACT_URL("emailContact"), NEW_INSTANCE_URL("new"), UPDATE_INSTANCE_URL("update"), GET_INSTANCE_URL("resource"),
    	GRAPH_URL("graph"), FAKE_WORKFLOW_URL("fakeworkflow"), LIST_GRAPHS_URL("listGraphs"), CLAIM_URL("workflow/claim"),
    	RELEASE_URL("workflow/release"), TRANSITION_URL("workflow/push"), LIST_TRANSITIONS_URL("workflow/transitions"),
		LIST_RESOURCES_URL("workflow/resources");
    	
    	private String url;
    	
    	private RepositoryLocale(String url) {
    		this.url = url;
    	}
    	
    	public String getURL() {
    		return url;
    	}
    }
}
