/**
 * The eagle-i consortium
 * Harvard University
 * Nov 5, 2010
 */
package org.eaglei.services.repository;

import java.io.IOException;
import java.io.InputStream;
import java.io.StringWriter;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpException;
import org.apache.commons.httpclient.HttpMethod;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.URIException;
import org.apache.commons.io.IOUtils;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.services.repository.RepositoryProviderException.RepositoryProviderExceptionType;

/**
 * @author Daniela Bourges-Waldegg
 *
 */
public class ProviderUtils {
	private static final Log logger = LogFactory.getLog( ProviderUtils.class );

	public static String getStringFromInputStream(final InputStream in) throws IOException {
		if ( in == null ) {
			return null;
		}
		final StringWriter writer = new StringWriter();
		// encoding needs to be explicitly set
		IOUtils.copy( in, writer, "UTF-8" );
		return writer.toString();
	}
	
	/**
	 * Convenience method for dealing with the most general cases of http request/response 
	 * 
	 * @param client
	 * @param method
	 * @return the string returned by the repo
	 * @throws HttpException
	 * @throws IOException
	 * @throws RepositoryProviderException
	 * @throws NotFoundException 
	 * @throws UnauthorizedException 
	 * @throws ConflictException 
	 * @throws InternalServerErrorException 
	 */
	public static String getHttpResponse(final HttpClient client, final HttpMethod method) throws RepositoryProviderException {
		int status = 0;

		if ( client == null ) {
			logger.warn( "trying to get response with null client" );
			throw new RepositoryProviderException("trying to get response with null client", RepositoryProviderExceptionType.INVALID_SESSION);
		}
		
		String originalPath = null;
		try {
			originalPath = method.getURI().toString();
		} catch (URIException e2) {
			logger.error(e2);
		}
		String responseBody;
		try {
			status = client.executeMethod( method );
			responseBody = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
		} catch (final HttpException e) {
			throw new RepositoryProviderException(e);
		} catch (IOException e1) {
			throw new RepositoryProviderException(e1);
		} finally {
			method.releaseConnection();
		}

		//TODO: Are any other status code considered successes?
		if ( status == HttpStatus.SC_OK || status == HttpStatus.SC_CREATED ) {
			//logger.info( "Request succeded with status: " + status );
			return responseBody;
		}
		// throw the appropriate exception
		RepositoryProviderExceptionType type = RepositoryProviderExceptionType.getTypeFromStatus(status);
		if (type != null) {
			if (type == RepositoryProviderExceptionType.MOVED_PERMANENTLY || type == RepositoryProviderExceptionType.MOVED_TEMPORARILY) {
				StringBuilder strBuild = new StringBuilder();
				strBuild.append(type==RepositoryProviderExceptionType.MOVED_TEMPORARILY?"Temporary":"Permanent");
				strBuild.append(" redirect encountered for url: '");
				strBuild.append(originalPath);
				strBuild.append("' should be: '");
				strBuild.append(method.getResponseHeader("Location").getValue());
				strBuild.append("'");
				logger.warn(strBuild.toString());
			}
			logger.error(type.getMessage());
			throw new RepositoryProviderException(type);
		}
		else {
			String message = "Operation failed with status: " + status + " response: " + responseBody; 
			logger.error(message);
			throw new RepositoryProviderException(message);
		}
	}
}
