package org.eaglei.solr;

import java.io.IOException;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.index.CorruptIndexException;
import org.apache.lucene.index.IndexWriter;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.ScoreDoc;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.LockObtainFailedException;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.ResourceChangeEvent;
import org.eaglei.search.harvest.ResourceChangeListener;

public abstract class AbstractLuceneIndexerNew implements ResourceChangeListener {

    private static final Log logger = LogFactory.getLog(AbstractLuceneIndexer.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    private static final DateFormat dateFormat = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");

    // A debug label for this indexer
    protected String indexerLabel;
    private int count;
    protected EIOntModel eiOntModel;
    protected Analyzer analyzer;
    protected Directory directory;
	// Flag indicating, don't bother querying the index if it's empty
    private boolean indexEmpty;
    
    // In memory cache of Documents that have been created or changed
    private Map<String, Map<String,Document>> mapFieldToKeyToDocument;
    // In memory cache of Documents that have been deleted
    private Map<String, Set<String>> mapFieldToDeletedKeySet;
    private Set<EIEntity> resourceRoots = new HashSet<EIEntity>();
    private Set<EIEntity> setInsitutionsWithInitialData = new HashSet<EIEntity>();
    /**
     * Cache of type URIs for instances that should be flattened
     * into their referencing instance.
     */
    private Set<EIURI> flattenTypeURIs = new HashSet<EIURI>();
           
    public AbstractLuceneIndexerNew(final String indexerLabel, final EIOntModel eiOntModel, final Analyzer analyzer, final Directory directory) {
        this.indexerLabel = indexerLabel;
        this.eiOntModel = eiOntModel;
        this.analyzer = analyzer;
        this.directory = directory;
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
			resourceRoots.add(c.getEntity());
		}
		initFlattenCache();
		this.indexEmpty = true;  // assume it is empty on construction (for now)
    }
    
    private void initFlattenCache() {
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_EMBEDDED_CLASS)) {
			addFlattenClass(c);
		}    	
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_SEARCH_FLATTEN)) {
			addFlattenClass(c);
		}    	
    }
    
    private void addFlattenClass(EIClass c) {
    	flattenTypeURIs.add(c.getEntity().getURI());
    	for (EIClass sub : eiOntModel.getSubClasses(c.getEntity().getURI())) {
        	flattenTypeURIs.add(sub.getEntity().getURI());
    	}
    }
    
    protected boolean isFlattenClass(EIURI uri) {
    	return flattenTypeURIs.contains(uri);
    }

    protected IndexWriter getWriter() throws CorruptIndexException, LockObtainFailedException, IOException {
    	return new IndexWriter(directory, analyzer, IndexWriter.MaxFieldLength.LIMITED);
    }

    @Override
    public void onChangeStreamStart(EIEntity institution) {
        logger.debug(indexerLabel + ": onChangeStreamStart: " + institution.getLabel());
        count = 0;
    }

    @Override
    public void onChangeEvent(ResourceChangeEvent event) {
    	assert (event != null) : indexerLabel + ": Null change event notification";
    	count++;
    	if (count % 500 == 0) {
    		logger.debug(indexerLabel + ": Received " + count + " change events...");
    	}
    }

    @Override
    public boolean onChangeStreamEnd(EIEntity institution, Date lastModifiedDate) {
        logger.debug(indexerLabel + ": onChangeStreamEnd: " + institution.getLabel() + "   num change events " + count + 
        		" last modifed: " + dateFormat.format(lastModifiedDate));
        return commitDocumentCache(institution);
    }
    
    /**
     * Commits the current in-memory Document cache to the index.
     */
    protected boolean commitDocumentCache(EIEntity institution) {
    	// Only return true when commit completes
    	boolean success = false;
    	IndexWriter iwriter = null;
        try {
            iwriter = getWriter();
    		// Perform deletes
        	if (mapFieldToDeletedKeySet != null && !indexEmpty) {
            	for (String field : mapFieldToDeletedKeySet.keySet()) {
            		Set<String> setDeletedKeys = mapFieldToDeletedKeySet.get(field);
    	            for (String key : setDeletedKeys) {
	            		deleteDocumentFromIndex(iwriter, field, key);    	            	
    	            }
            	}        		
        	}
        	// Perform adds/updates
        	if (mapFieldToKeyToDocument != null) {
	        	for (String field : mapFieldToKeyToDocument.keySet()) {
	        		Map<String,Document> mapKeyToDocument = mapFieldToKeyToDocument.get(field);
		            for (String key : mapKeyToDocument.keySet()) {
		                if (setInsitutionsWithInitialData.contains(institution)) {
		                	// Only log this message after initial dataset has been received.
		                	//logger.debug(indexerLabel + ": Updating " + key);
		                }
		                Document doc = mapKeyToDocument.get(key);
		            	if (!indexEmpty) {
		            		// Delete documents corresponding to this key
		            		// before adding them.
		            		deleteDocumentFromIndex(iwriter, field, key);
		            	}
		            	iwriter.addDocument(doc);
		            }
	        	}
	            if (institution != null && mapFieldToKeyToDocument.size() > 0) {
	            	indexEmpty = false;
	            	setInsitutionsWithInitialData.add(institution);
	            }
        	}
        } catch (Exception e) {
            logger.error(indexerLabel + ": Error updating Documents: indexEmpty: " + indexEmpty, e);
        } finally {
            try {
            	// Close commits changes to the index
            	if (iwriter != null) {
            		iwriter.close();
            	}
				success = true;
				logger.debug(indexerLabel + ": Wrote updated Documents to index.");
	            //logger.debug(indexerLabel + ": Wrote " + mapKeyToDocument.size() + " updated Documents to index.");
			} catch (Exception e) {
	            logger.error(indexerLabel + ": Error closing IndexWriter after Document update", e);
	        	try {
	        		// Be sure to leave the directory unlocked after close, particularly after
	        		// an exception.  This should generally only be done on exception.
	        		// And the index writer should not be reused.
					if (IndexWriter.isLocked(directory)) {
						IndexWriter.unlock(directory);
					}
				} catch (IOException e2) {
		            logger.error(indexerLabel + ": Error unlocking after update", e2);
				}
			} finally {
		        iwriter = null;
		        mapFieldToKeyToDocument = null;
		        mapFieldToDeletedKeySet = null;
			}
        }
        return success;
    }
    
    /**
     * Gets the Document associated with this key.
     * If the document associated with this key has been previously
     * created, it is guaranteed to be in cache.
     * 
     * @param key
     * @return
     */
    protected Document getDocument(final String keyField, final String key) {
    	Document doc;
    	// First check cache
    	if (mapFieldToKeyToDocument != null) {
			Map<String,Document> mapKeyToDocument = mapFieldToKeyToDocument.get(keyField);
			if (mapKeyToDocument != null) {
		    	doc = mapKeyToDocument.get(key);
		        if (doc != null) {
		            return doc;
		        }
			}
    	}
        // Then the index
    	if (!indexEmpty) {
    		doc = getDocumentFromIndex(keyField, key);
            if (doc != null) {
            	// Ensure the document is brough into cache, so that any edits will be committed
            	setDocument(keyField, key, doc);
                return doc;
            }  
    	}
        return null;
    }
    
    protected void setDocument(final String keyField, final String key, final Document doc) {
		if (mapFieldToKeyToDocument == null) {
	        mapFieldToKeyToDocument = new HashMap<String, Map<String, Document>>();
		}
		Map<String,Document> mapKeyToDocument = mapFieldToKeyToDocument.get(keyField);
		if (mapKeyToDocument == null) {
			mapKeyToDocument = new HashMap<String,Document>();
			mapFieldToKeyToDocument.put(keyField, mapKeyToDocument);
		}
    	mapKeyToDocument.put(key, doc);    	
    }
    
	protected Document getDocumentFromIndex(final String keyField, final String key) {
		try {
	        final TermQuery keyQuery = new TermQuery(new Term(keyField, key));
	        
	        // TODO shouldn't have to new one up
	        final IndexSearcher searcher = new IndexSearcher(directory, true);
	        //searcher.setDefaultFieldSortScoring(false, false);
	        final TopDocs docs = searcher.search(keyQuery, 1);
	        if (docs.totalHits == 0) {
	            //logger.error("Did not find " + key + " in search index");
	            return null;
	        }
	        assert(docs.scoreDocs.length == 1) : "Unique document key wasn't unique: " + key;
	        return searcher.doc(docs.scoreDocs[0].doc);
		} catch (IOException e) {
			logger.error(e);
			return null;
		}
	}
    
    protected void deleteDocument(final String keyField, final String key) {
    	// Delete from document cache
    	if (mapFieldToKeyToDocument != null) {
			Map<String,Document> mapKeyToDocument = mapFieldToKeyToDocument.get(keyField);
			if (mapKeyToDocument != null) {
				mapKeyToDocument.remove(key);
			}
    	}
		// Store key for delete from index on cache commit
    	if (!indexEmpty) {
    		if (mapFieldToDeletedKeySet == null) {
    	        mapFieldToDeletedKeySet = new HashMap<String, Set<String>>();
    		}
    		Set<String> deletedKeySet = mapFieldToDeletedKeySet.get(keyField);
    		if (deletedKeySet == null) {
    			deletedKeySet = new HashSet<String>();
    			mapFieldToDeletedKeySet.put(keyField, deletedKeySet);
    		}
    		deletedKeySet.add(key);
    	}
    }
    
    private void deleteDocumentFromIndex(final IndexWriter iwriter, final String keyField, final String key) {
    	try {
	        final TermQuery keyQuery = new TermQuery(new Term(keyField, key));
	        iwriter.deleteDocuments(keyQuery);        
		} catch (IOException e) {
			logger.error(indexerLabel + ": Unexpected error during delete", e);
		}
    }
    
    public void optimize() {
        if (indexEmpty) {
        	return;
        }
    	logger.debug(indexerLabel + ": Optimizing index...");
    	IndexWriter iwriter = null;
    	try {
    		iwriter = getWriter();
			iwriter.optimize();
		} catch (Exception e) {
			logger.error(indexerLabel + ": Unexpected error during optimize", e);
		} finally {
            try {
            	// Close commits changes to the index
				iwriter.close();
	        	logger.debug(indexerLabel + ": Optimize complete.");
			} catch (Exception e) {
	            logger.error(indexerLabel + ": Error closing IndexWriter after optimize", e);
	        	try {
	        		// Be sure to leave the directory unlocked after close, particularly after
	        		// an exception.  This should generally only be done on exception.
	        		// And the index writer should not be reused.
					if (IndexWriter.isLocked(directory)) {
						IndexWriter.unlock(directory);
					}
				} catch (IOException e2) {
		            logger.error(indexerLabel + ": Error unlocking after optimize", e2);
				}
			}
		}
    }
    
}
