package org.eaglei.solr.harvest;

import java.util.ArrayList;
import java.util.List;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.harvest.DataHarvester;
import org.eaglei.search.harvest.PollingDataHarvester;
import org.eaglei.search.harvest.ResourceChangeListener;
import org.eaglei.services.InstitutionRegistry;

/**
 * Extension of AbstractStreamHarvester that uses the Repository /harvest API (see /harvest REST API spec for details).
 * @author tbashor
 */
// TODO retrieve inferred triples as well (currently inferred triples are being computed locally by indexer)
public final class MultiDataSourceStreamHarvester implements PollingDataHarvester {

    private class MultiDataSourceHarvestPoller extends Thread {
        
    	MultiDataSourceHarvestPoller() {
            super("MultiDataSourceHarvestPoller");
            setPriority(Thread.MIN_PRIORITY);
            setDaemon(true);
        }

        @Override
        public void run() {
            while (true) {
        		for (DataHarvester harvester : harvesters) {
	                try {
	                    harvester.harvest();
	                } catch (Throwable t) {
	                    logger.error("Unexpected error during harvest", t);
	                }

	                try {
	                    //TODO make configurable
	                    Thread.sleep(2000);
	                } catch (InterruptedException ie) {
	                    // swallow
	                }
        		}
        		if (!hasInitialData) {
        			boolean allHaveData = true;
            		for (DataHarvester harvester : harvesters) {
            			if (!harvester.hasInitialData()) {
            				allHaveData = false;
            				break;
            			}
            		}
            		if (allHaveData) {
            			// For now, only optimize after the initial
            			// bootstrap indexing round.
            			notifyOptimize();
            			hasInitialData = true;
            		}
        		}
            }
        }
        
    }
    
    private static final Log logger = LogFactory.getLog(MultiDataSourceStreamHarvester.class);
    
	private boolean hasInitialData = false;
    private final List<DataHarvester> harvesters;
    private MultiDataSourceHarvestPoller pollingThread;
    private final List<ResourceChangeListener> listeners = new ArrayList<ResourceChangeListener>();
        
    public MultiDataSourceStreamHarvester() {
    	this.harvesters = new ArrayList<DataHarvester>();
    }
    
    public void addDataHarvester(DataHarvester harvester) {
    	harvesters.add(harvester);
    }

	@Override
	public void addChangeListener(ResourceChangeListener listener) {
		for (DataHarvester harvester : harvesters) {
			harvester.addChangeListener(listener);
		}
		listeners.add(listener);
	}

    @Override 
    public boolean hasInitialData() {
		return hasInitialData;
    }
    
	@Override
	public void startPolling() {
        if (pollingThread != null) {
            return;
        }
        pollingThread = new MultiDataSourceHarvestPoller();
        pollingThread.start();
	}

	@Override
	public void harvest() {
    	for (DataHarvester harvester : harvesters) {
    		harvester.harvest();
    	}
	}

    protected void notifyOptimize() {
    	for (ResourceChangeListener listener : listeners) {
            try {
        		listener.optimize();
            } catch (Throwable t) {
                logger.error("Unexpected error during optimize notification", t);
            }
    	}
    }
    
}