package org.eaglei.solr.harvest;

import java.io.FileWriter;
import java.io.IOException;
import java.io.InputStream;
import java.io.Reader;
import java.text.DateFormat;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.Locale;
import java.util.SimpleTimeZone;
import java.util.TimeZone;

import org.apache.commons.httpclient.HttpClient;
import org.apache.commons.httpclient.HttpStatus;
import org.apache.commons.httpclient.SimpleHttpConnectionManager;
import org.apache.commons.httpclient.methods.PostMethod;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.services.repository.ProviderUtils;
import org.eaglei.services.repository.RepositoryHttpConfig;
import org.eaglei.services.repository.RepositoryHttpConfig.RepositoryLocale;

/**
 * Extension of AbstractStreamHarvester that uses the Repository /harvest API (see /harvest REST API spec for details).
 * @author tbashor
 */
// TODO retrieve inferred triples as well (currently inferred triples are being computed locally by indexer)
public final class RepositoryStreamHarvester extends AbstractStreamHarvester {

    private static final Log logger = LogFactory.getLog(RepositoryStreamHarvester.class);
    private static final boolean DEBUG = logger.isDebugEnabled();
    
    private String lastModifiedStr = null;
    private Date lastModifiedDate = null;
    
    private static final DateFormat format = new SimpleDateFormat("yyyy-MM-dd'T'HH:mm:ss.SSS");
    private static final SimpleDateFormat preciseHTTPDate =
        new SimpleDateFormat("EEE, dd MMM yyyy HH:mm:ss.SSS zzz");
    static {
        format.setTimeZone(TimeZone.getTimeZone("GMT"));
        preciseHTTPDate.setCalendar(new GregorianCalendar(new SimpleTimeZone(0, "GMT"), Locale.getDefault()));
    }
    
    private final RepositoryHttpConfig repoConfig;
    
    /*
     * HttpClient that is reused. It will reuse the underlying connection or recreate if it gets dropped.
     */
    private HttpClient httpclient;
    
    private String harvestURL;
    
    /**
     * Creates a new RepositoryHarvester.
     */
    /*
    public RepositoryStreamHarvester(final EIOntModel eiOntModel, final InstitutionRegistry institutionRegistry,
            final ResourceChangeListener listener) {
        this(eiOntModel, institutionRegistry.getInstitution(), institutionRegistry, listener);
    }
    */

    /**
     * Creates a new RepositoryHarvester.
     */
    public RepositoryStreamHarvester(final EIOntModel eiOntModel, final EIEntity institution,
            final InstitutionRegistry institutionRegistry) {
        super(eiOntModel, institution);
  
        this.repoConfig = institutionRegistry.getRepositoryHttpConfig(institution.getURI().toString());

        httpclient = RepositoryHttpConfig.createHttpClient(repoConfig.getSearchUsername(), repoConfig.getSearchPassword());
        // Assume single threaded use of this client
        // Configure to try to keep connection open between uses.
        httpclient.setHttpConnectionManager(new SimpleHttpConnectionManager(false));
        
    	harvestURL = repoConfig.getFullRepositoryUrl(RepositoryLocale.HARVEST_URL);

        if (DEBUG) {
            logger.debug("Created repository harvester for " + institution.getLabel() + " url: " + harvestURL);
        }
    }
    
    public String getHarvestInfo() {
    	if (lastModifiedDate != null) {
    		return "Dataset:   " + formatWithTZ.format(lastModifiedDate) + "    [" + harvestURL + "]";
    	} else {
    		return harvestURL;
    	}
    }

    /**
     * Executes the repository /harvest API and notifies listeners of change information.
     */    
    public void harvest() {        
        
        //if (DEBUG) {
        //	logger.debug("Making harvest request to " + repoConfig.getHarvestUrl());
        //}
        
        // create PostMethod and set parameters
        final PostMethod method = new PostMethod(harvestURL);
        method.setParameter("view", RepositoryHttpConfig.PUBLISHED_VIEW);
        method.setParameter("detail", "full");
        
        // if we have a fromTime value, set that
        if (lastModifiedDate != null) {
            method.setParameter("after", format.format(lastModifiedDate) + "Z"); // GMT time 
        } 
        
        InputStream is = null;
        Reader in = null;
        FileWriter fw = null;
        try {
            // execute the /harvest call    
            int status = httpclient.executeMethod(method);
            if ( status != HttpStatus.SC_OK  ) {
                if (status == HttpStatus.SC_UNAUTHORIZED) {
                    // Repo server is probably not up yet.
                    //logger.warn( "harvest request status: Unauthorized");
                } else {
                    final String response = ProviderUtils.getStringFromInputStream( method.getResponseBodyAsStream() );
                    logger.error(harvestURL + ": failed with status: " + status + "\r\n" + response);
                }
                return;
            }

            String newLastModifiedStr = method.getResponseHeader("X-Precise-Last-Modified").getValue();
            if (newLastModifiedStr.equals(lastModifiedStr)) {
                return; // no changes
            }
            Date newLastModifiedDate;
            try {
            	newLastModifiedDate = preciseHTTPDate.parse(newLastModifiedStr);
			} catch (ParseException e) {
                logger.error(harvestURL + ": Error parsing X-Precise-Last-Modified " + newLastModifiedStr);
                return;
			}
            
            is = method.getResponseBodyAsStream();
            
            /*
            in = new InputStreamReader(is);
            String fileName = System.getProperty(InstitutionRegistry.EAGLEI_TIER_PROPERTY) + "_" + System.getProperty(InstitutionRegistry.EAGLEI_SUBDOMAIN_PROPERTY) + ".txt";
            fw = new FileWriter(fileName);
            if (DEBUG) {
            	logger.debug("Writing harvest stream to " + fileName);
            }
            int c;
            while ((c = in.read()) != -1) {
            	fw.write(c);
            }
            fw.flush();
            if (DEBUG) {
            	logger.debug("Finished writing to " + fileName);
            }
            */
            
            notifyChangeStreamStart();
            int count = generateResourceChangeEvents(is);
			if (notifyChangeStreamEnd(newLastModifiedDate)) {
				if (count > 0) {
					// phantom harvest response SEARCH-441
					// if event count is zero don't update last modified
					lastModifiedStr = newLastModifiedStr;
					lastModifiedDate = newLastModifiedDate;
					if (!hasInitialData && count > 0) {
						hasInitialData = true;
		                logger.debug(institution.getLabel() + " initial data has been harvested.");
					}
				} else {
					// If event count is zero don't update last modified.  See SEARCH-441
					// dump debug
					StringBuilder buf = new StringBuilder("Response with zero events\r\n");
					buf.append("Request url: " + method.getURI() + "\r\n");
					buf.append("Request param: after: " + format.format(lastModifiedDate) + "Z" + "\r\n");
					buf.append("Reponse header: X-Precise-Last-Modified: " + newLastModifiedStr + "\r\n");
	                logger.error(buf.toString());
				}
			}
        } catch (IOException e) {
        	logger.debug("Unexpected error in RepositoryStreamHarvester " + harvestURL, e);        	
        } finally {
            method.releaseConnection();
            if (is != null) {
                try {
					is.close();
				} catch (IOException e) {}
            }
            if (in != null) {
            	try {
					in.close();
				} catch (IOException e) {}
            }
            if (fw != null) {
            	try {
					fw.close();
				} catch (IOException e) {}
            }
        }        
    }

}