package org.eaglei.solr.suggest;

import java.io.File;
import java.io.IOException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.store.Directory;
import org.apache.lucene.store.FSDirectory;
import org.apache.lucene.store.RAMDirectory;
import org.eaglei.lexical.lucene.AutoSuggestIndexAnalyzer;
import org.eaglei.lexical.lucene.AutoSuggestQueryAnalyzer;
import org.eaglei.model.EIOntModel;
import org.eaglei.search.harvest.PollingDataHarvester;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.solr.harvest.DataHarvesterConfig;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;

@Configuration
@Import(DataHarvesterConfig.class)
public class LuceneAutoSuggestConfig {

	private static final Log logger = LogFactory.getLog(LuceneAutoSuggestConfig.class);

    private @Autowired EIOntModel eagleiOntModel;
    private @Autowired InstitutionRegistry institutionRegistry;
    private @Autowired PollingDataHarvester dataHarvester;
    
    public @Bean Directory autoSuggestDirectory() {
        try {
            String indexDir = System.getProperty("org.eaglei.lucene.dir");
            
            // if the index directory property is not null, create an FSDirectory for that
            if (indexDir != null) {
            	indexDir += "/suggest";
            	indexDir += "/" + institutionRegistry.getLocalNodeId();
                final File dir = new File(indexDir);
                logger.debug("Using Lucene index directory: " + dir.getAbsolutePath());
                if (dir.exists()) {
                	// Delete existing dir
                	// TODO parameterize
                	File[] contents = dir.listFiles();
                	if (contents != null) {
                		for (File f : contents) {
                        	if (!f.delete()) {
        	                    throw new RuntimeException("Error deleting lucene directory: " + f.getAbsolutePath());
                        	}
                		}
                	}
                	if (!dir.delete()) {
	                    throw new RuntimeException("Error deleting lucene directory: " + dir.getAbsolutePath());
                	}
                    logger.debug("Deleted Lucene index directory: " + dir.getAbsolutePath());
                }
                if (!dir.mkdirs()) {
                    throw new RuntimeException("Error creating lucene directory: " + dir.getAbsolutePath());                	
                }
                return FSDirectory.open(dir);
            } else {
	            Directory ram = new RAMDirectory();
	            logger.debug("Using Lucene RAMDirectory " + ram.toString());
	            return ram;
            }
		} catch (IOException e) {
	        logger.error("Exception getting Lucene directory, using RAMDirectory. ", e);
	        return new RAMDirectory();
		}
    }

    public @Bean Analyzer autoSuggestIndexAnalyzer() {
        return new AutoSuggestIndexAnalyzer();
    }
    
    public @Bean Analyzer autoSuggestQueryAnalyzer() {
        return new AutoSuggestQueryAnalyzer();
    }
    
    public @Bean LuceneDataSuggestIndexer autoSuggestIndexer() {
    	LuceneDataSuggestIndexer indexer = new LuceneDataSuggestIndexer(eagleiOntModel, 
        		autoSuggestIndexAnalyzer(), autoSuggestDirectory());
        dataHarvester.addChangeListener(indexer);
        return indexer;
    }

    public @Bean LuceneDataSuggestProvider autoSuggestProvider() throws IOException {    
        return new LuceneDataSuggestProvider(autoSuggestQueryAnalyzer(), 
        		autoSuggestIndexAnalyzer(), autoSuggestDirectory());
    }    

}
