package org.eaglei.solr.suggest;

import java.io.IOException;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.lucene.analysis.Analyzer;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.BooleanClause;
import org.apache.lucene.search.BooleanQuery;
import org.apache.lucene.search.IndexSearcher;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.apache.lucene.store.Directory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIObjectProperty;
import org.eaglei.model.EIOntConstants;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.harvest.ResourceChangeEvent;
import org.eaglei.search.harvest.ResourceChangeListener;
import org.eaglei.solr.AbstractLuceneIndexerNew;

public class LuceneDataSuggestIndexer extends AbstractLuceneIndexerNew implements LuceneDataSuggestIndexSchema,ResourceChangeListener {

    private static final Log logger = LogFactory.getLog(LuceneDataSuggestIndexer.class);

    private static final EIURI DOCUMENT_URI = EIURI.create("http://purl.obolibrary.org/obo/IAO_0000310");
    private static final EIURI PROTOCOL_URI = EIURI.create("http://purl.obolibrary.org/obo/OBI_0000272");
    private Set<EIEntity> categoryRoots = new HashSet<EIEntity>();
        
    public LuceneDataSuggestIndexer(final EIOntModel eiOntModel, final Analyzer analyzer, final Directory directory) {
    	super("DataSuggestIndexer", eiOntModel, analyzer, directory);
		for (EIClass c : eiOntModel.getClassesInGroup(EIOntConstants.CG_DATA_MODEL_CREATE)) {
			categoryRoots.add(c.getEntity());
		}
    }
    
    @Override
    public void onChangeEvent(ResourceChangeEvent event) {
    	super.onChangeEvent(event);
    	if (event.isDelete()) {
    		deleteResourceInstance(event);
    	} else {
    		indexResourceInstance(event);
    	}
    }
    
    /*
     * Note that this could be a new resource or an update.
     */
    private void indexResourceInstance(final ResourceChangeEvent event) {
    	//logger.debug("Index resource " + event.getEntity() + "  Type: " + event.getType());

    	EIEntity resourceEntity = event.getEntity();
    	String resourceURIStr = resourceEntity.getURI().toString();
    	EIEntity typeEntity = event.getType();
    	if (eiOntModel.isSubClass(DOCUMENT_URI, typeEntity.getURI())
    			&& !eiOntModel.isSubClass(PROTOCOL_URI, typeEntity.getURI())) {
    		// Exclude non-protocol document titles from autosuggest
    		//logger.debug("Excluding from suggest: " + resourceEntity + " of type: " + typeEntity);
    		return;
    	}

    	String categoryURIStr = getCategory(typeEntity);
    	
    	Document doc = 
    		createEntity(true, resourceURIStr, resourceEntity.getLabel(), categoryURIStr);
    	// Add the resource label in the resource's category
		String entitySuggestLabel = doc.get(FIELD_ENTITY_LABEL);
    	addLabelReference(entitySuggestLabel, categoryURIStr, doc, null);
    	// Add reference to the asserted type
    	addClassReferenceWithSubsumption(typeEntity.getURI().toString(), doc, resourceURIStr, categoryURIStr);
        
        // Add the object property references.
        // Will create stub documents for any instance references
        // that we don't have data for yet.
        for (EIObjectProperty prop : event.getObjectProperties()) {
        	for (EIURI valueURI : event.getObjectProperty(prop)) {
        		addPropertyReference(valueURI, doc, resourceURIStr, categoryURIStr);
        	}
        }
        
        // special meta property for resource provider
        if (event.getProvider() != null) {
        	EIURI resourceProviderURI = event.getProvider();
        	addInstanceReference(resourceProviderURI.toString(), doc, resourceURIStr, categoryURIStr);
        }
        
        setDocument(FIELD_ENTITY_URI, resourceURIStr, doc);
    }
    
    /*
     * Ensures that an entity record exists for the given class or instance entity.
     * Class records will have uri and label.
     * Instance records will have uri, label, and optionally category
     * An instance record with no label is a stub record.
     * An instance record with no category is an instance that lies outside
     * the categorized type hierarchy.
     * 
     * Other code should always retrieve the entity label from the Document.
     * Any normalization behavior (e.g. lower casing) will be performed here.
     */
    private Document createEntity(boolean isInstance, String uriStr, String label, String categoryURIStr) {    	
    	// Check if a doc for this uri already exists
    	Document previousEntityDoc = getDocument(FIELD_ENTITY_URI, uriStr);
    	
        // Create a new Lucene document for the entity
    	// Either a new class, a new instance, a new stub instance, or a stub instance replacemnt.
        final Document entityDoc = new Document();
        entityDoc.add(new Field(FIELD_ENTITY_URI, uriStr, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        entityDoc.add(new Field(FIELD_ENTITY_IS_INSTANCE, Boolean.toString(isInstance), Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        if (label != null) {
        	label = label.trim().toLowerCase();
        	entityDoc.add(new Field(FIELD_ENTITY_LABEL, label, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        }
        if (categoryURIStr != null) {
        	entityDoc.add(new Field(FIELD_ENTITY_INSTANCE_CATEGORY, categoryURIStr, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        }
        // Store the new or update entity document
        setDocument(FIELD_ENTITY_URI, uriStr, entityDoc);

    	if (previousEntityDoc != null && label != null) {
    		assert(isInstance);
    		// This is a stub instance which now has a known label.
    		// If other instances have referenced this instance as a property value,
    		// we now need to record usage of this instance label using the category of the 
    		// referencing instance.
    		String[] referencingURIStrs = previousEntityDoc.getValues(FIELD_ENTITY_REFERENCED_BY);
    		if (referencingURIStrs != null) {
    			for (String referencingURIStr : referencingURIStrs) {
    				// Record the reference to this instance's label
    				// (equivalent to what would have happened in addInstanceReference)
    		    	Document referencingDocument = getEntityDocument(referencingURIStr);
    		    	if (referencingDocument == null) {
    		    		// Check for referenced_by value that is not valid
    		    		logger.error(indexerLabel + ": " + label + " : " + uriStr + " has referenced_by uri that doesn't exist: " + referencingURIStr);
    		    		continue;
    		    	}
    		    	String referencingCategoryURIStr = referencingDocument.get(FIELD_ENTITY_INSTANCE_CATEGORY);
    		    	addInstanceReference(uriStr, referencingDocument, referencingURIStr, referencingCategoryURIStr);
    			}
        	}
    	}

        return entityDoc;
    }
    
    private void addPropertyReference(EIURI referencedURI, 
    		Document referencingDocument, String referencingURIStr, String referencingCategoryURIStr) {
    	String referencedURIStr = referencedURI.toString();
        if (eiOntModel.isModelClassURI(referencedURIStr)) {
        	// Value is a model class
        	addClassReferenceWithSubsumption(referencedURIStr, referencingDocument, referencingURIStr, referencingCategoryURIStr);
        } else {
        	// Value is a resource
        	addInstanceReference(referencedURIStr, referencingDocument, referencingURIStr, referencingCategoryURIStr);
        }
    }
    
    private void addClassReferenceWithSubsumption(String classURIStr, 
    		Document referencingDocument, String referencingURIStr, String referencingCategoryURIStr) {
    	EIURI classURI = EIURI.create(classURIStr);
    	addClassReference(classURI, referencingDocument, referencingURIStr, referencingCategoryURIStr);
    	// Add reference to the inferred supertypes
        List<EIClass> resourceSuperClasses = eiOntModel.getSuperClasses(classURI);
        for (EIClass superClass : resourceSuperClasses) {
        	addClassReference(superClass.getEntity().getURI(), referencingDocument, referencingURIStr, referencingCategoryURIStr);
        }    	
    }
    
	private void addClassReference(EIURI classURI, 
			Document referencingDocument, String referencingURIStr, String referencingCategoryURIStr) {
		String classSuggestLabel;
		// Ensure that there exists an entity record for the class
    	Document classDocument = getEntityDocument(classURI.toString());
    	if (classDocument == null) {
    		EIEntity classEntity = eiOntModel.getClass(classURI).getEntity();
    		classDocument = createEntity(false, classEntity.getURI().toString(), classEntity.getLabel(), null);
    		classSuggestLabel = classDocument.get(FIELD_ENTITY_LABEL);
    		for (String synonym : eiOntModel.getLabels(classURI)) {
    			synonym = synonym.trim().toLowerCase();
    			if (synonym.equals(classSuggestLabel)) {
    				continue;
    			}
    			classDocument.add(new Field(FIELD_ENTITY_SYNONYM, synonym, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
    		}
    	}
    	// Record the usage of the class label and its synonyms in the suggest index
		classSuggestLabel = classDocument.get(FIELD_ENTITY_LABEL);
    	addLabelReference(classSuggestLabel, referencingCategoryURIStr, referencingDocument, classDocument);
    	String[] synonyms = classDocument.getValues(FIELD_ENTITY_SYNONYM);
    	for (String synonym : synonyms) {
    		addLabelReference(synonym, referencingCategoryURIStr, referencingDocument, classDocument);
    	}
    }
    
    private void addInstanceReference(String referencedInstanceURIStr, 
    		Document referencingDocument, String referencingURIStr, String referencingCategoryURIStr) {
    	// Ensure that there exists an entity record for the class.
    	Document referencedInstanceDoc = getEntityDocument(referencedInstanceURIStr);
    	if (referencedInstanceDoc == null) {
        	// Create a stub entity if instance doesn't exist yet.
    		referencedInstanceDoc = createEntity(true, referencedInstanceURIStr, null, null);
    	}
    	// Record the reference from the instance to the class
    	addReferencedBy(referencedInstanceDoc, referencingURIStr);
    	// Record the usage of the class label in the suggest index
		// Note that may be a stub and label may still be null.
		String instanceLabel = referencedInstanceDoc.get(FIELD_ENTITY_LABEL);
    	if (instanceLabel != null) {
    		addLabelReference(instanceLabel, referencingCategoryURIStr, referencingDocument, null);
    	}
    }
    
    /*
     * Adds the given URI as referencing this Document.
     * Note that there a given URI may reference this document multiple times
     * and therefore would have multiple fields representing each reference.
     */
    private void addReferencedBy(Document doc, String referencedByURIStr) {
        Field field = new Field(FIELD_ENTITY_REFERENCED_BY, referencedByURIStr, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS);
        doc.add(field);
    }
    
    /*
     * Removes a referenced by field with this URI as the value.
     */
    private void removeReferencedBy(Document doc, String referencedByURIStr) {
    	String[] existingRefByStrs = doc.getValues(FIELD_ENTITY_REFERENCED_BY);
		doc.removeFields(FIELD_ENTITY_REFERENCED_BY);
		boolean foundOne = false;
    	for (String existingRefByStr : existingRefByStrs) {
    		if (!foundOne && existingRefByStr.equals(referencedByURIStr)) {
    			foundOne = true;
    			continue;
    		}
    		// Add back all the others
    		addReferencedBy(doc, existingRefByStr);
    	}
    }
    
    /*
     * Called when an instance document in some category uses a given label.
     * 
     * Reference could come via a direct relationship (the instance label), or via a class or 
     * other instance relationship.
     * The label will be added to the suggest document set (if it's not already there).
     * The category of the instance causing this label to be added is also recorded (the category 
     * may already be set on the label, if another instance previously caused this label to exist 
     * in the particular category).
     * 
     * Finally, the reference to the label key itself recorded back onto the instance itself.  This is for cleanup
     * purposes on resource deletion.
     * Likewise, the label key will be written on the class document (if a class is the source of the label).
     * Also to be utilized during clean up on resource deletion.
     */
    private void addLabelReference(String label, String instanceCategoryURIStr, 
    		Document instanceDocument, Document classDocument) {
    	assert(label.equals(label.trim().toLowerCase())) : "Probably called addLabel w/out getting the string from the entity Document";
        assert(instanceCategoryURIStr != null) : "Null category id";
    	Document suggestDocument = getDocument(FIELD_SUGGEST_LABEL_KEY, label);
    	if (suggestDocument == null) {
    		// Label has never been seen before, create a new one.
    		suggestDocument = new Document();
    		// not analyzed
            suggestDocument.add(new Field(FIELD_SUGGEST_LABEL_KEY, label, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
            // analyzed
            suggestDocument.add(new Field(FIELD_SUGGEST_LABEL_SEARCH, label, Field.Store.YES, Field.Index.ANALYZED, Field.TermVector.WITH_POSITIONS_OFFSETS));
    	}
    	
    	// Ensure that the instance category uri is represented in the category list
    	String[] categories = suggestDocument.getValues(FIELD_SUGGEST_INSTANCE_CATEGORY);
    	boolean alreadyHasCategory = false;
    	for (String existingCategory : categories) {
    		if (existingCategory.equals(instanceCategoryURIStr)) {
    			alreadyHasCategory = true;
    			break;
    		}
    	}
    	if (!alreadyHasCategory) {
            suggestDocument.add(new Field(FIELD_SUGGEST_INSTANCE_CATEGORY, instanceCategoryURIStr, Field.Store.YES, Field.Index.NOT_ANALYZED));
    	}
        
    	// Record the fact that this label was added to the suggest index because of
    	// the following instances
        instanceDocument.add(new Field(FIELD_ENTITY_LABEL_REFERENCE, label, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        if (classDocument != null) {
        	classDocument.add(new Field(FIELD_ENTITY_LABEL_REFERENCE, label, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
        }

        setDocument(FIELD_SUGGEST_LABEL_KEY, label, suggestDocument);
    }
    
    private Document getEntityDocument(String uriStr) {
    	return getDocument(FIELD_ENTITY_URI, uriStr);
    }
    
    private String getCategory(EIEntity typeEntity) {
    	if (categoryRoots.contains(typeEntity)) {
    		return typeEntity.getURI().toString();
    	} else {
	        List<EIClass> resourceSuperClasses = eiOntModel.getSuperClasses(typeEntity.getURI());
	        for (EIClass superClass : resourceSuperClasses) {
	        	if (categoryRoots.contains(superClass.getEntity())) {
	        		return superClass.getEntity().getURI().toString();
	        	}
	        }   
	        return UNKNOWN_CATEGORY;
    	}
    }

    private void deleteResourceInstance(final ResourceChangeEvent event) {
    	EIEntity resourceEntity = event.getEntity();
    	String uriStr = resourceEntity.getURI().toString();
    	Document instanceDoc = getDocument(FIELD_ENTITY_URI, uriStr);
    	if (instanceDoc == null) {
    		logger.debug(indexerLabel + ": Delete event for entity not found in index: " + uriStr);
    		return;
    	}
		String instanceLabel = instanceDoc.get(FIELD_ENTITY_LABEL);
		logger.debug(indexerLabel + ": Deleting: " + instanceLabel + " : " + uriStr);
    	// Remove our label reference from all referencing instances.
		if (instanceLabel != null) {
			String[] referencingURIStrs = instanceDoc.getValues(FIELD_ENTITY_REFERENCED_BY);
			if (referencingURIStrs != null) {
				for (String referencingURIStr : referencingURIStrs) {
					removeInstanceReference(instanceDoc, referencingURIStr);
				}
	    	}
		}
		// Save off our label reference list and category before deleting
		String[] labelReferences = instanceDoc.getValues(FIELD_ENTITY_LABEL_REFERENCE);
		String category = instanceDoc.get(FIELD_ENTITY_INSTANCE_CATEGORY);
		deleteDocument(FIELD_ENTITY_URI, uriStr);
    	// Remove the usage of the label by this referencing instance in the suggest index
		for (String labelReference : labelReferences) {
			removeLabelReference(labelReference, category, instanceDoc);
		}
    }
    
    private void removeInstanceReference(Document referencedDocument, String referencingURIStr) {
    	Document referencingDocument = getEntityDocument(referencingURIStr);
    	if (referencingDocument == null) {
    		logger.error("removeInstanceReference: referencing document does not exist: " + referencingURIStr);
    		return;
    	}
    	
    	// Remove the reference_by field
    	// (Is this actually necessary?)
    	removeReferencedBy(referencedDocument, referencingURIStr);
    	
		// Remove use of this label from referencing document.
		// Note that the referencing document could reference this instance multiple times,
		// or it could reference other things with this label.  So be sure to only remove
		// one usage per reference.
		String label = referencedDocument.get(FIELD_ENTITY_LABEL);
    	if (label == null) {
    		logger.warn("removeInstanceReference called for referenced document with null label: " + referencedDocument.get(FIELD_ENTITY_URI));   
    		return;
    	}
    	String[] labelReferences = referencingDocument.getValues(FIELD_ENTITY_LABEL_REFERENCE);
    	referencingDocument.removeFields(FIELD_ENTITY_LABEL_REFERENCE);
    	boolean foundOne = false;
		logger.debug("  Remove use of label " + label + " from referencing document: " + referencingDocument.get(FIELD_ENTITY_LABEL));
		for (String labelReference : labelReferences) {
			if (!foundOne && labelReference.equals(label)) {
				foundOne = true;
	    		logger.debug("     label use: " + labelReference + "  [removed]");
				continue;
			}
			referencingDocument.add(new Field(FIELD_ENTITY_LABEL_REFERENCE, labelReference, Field.Store.YES, Field.Index.NOT_ANALYZED_NO_NORMS));
    		logger.debug("     label use: " + labelReference);
		}
		
    	// Remove the usage of the label by this referencing instance in the suggest index.
		// Must be called after first removing the label from this document's LABEL_REFERENCE list.
    	String category = referencingDocument.get(FIELD_ENTITY_INSTANCE_CATEGORY);
    	if (category == null) {
    		logger.error("removeInstanceReference: category of referencing doc should not be null: " + referencingURIStr);
    		return;
    	}
		removeLabelReference(label, category, referencingDocument);
    }
   
    private void removeLabelReference(String label, String category, Document referencingDocument) {
		// See if there are any instances that reference this label, and are in this category
		if (!isLabelReference(label, category)) {
			// If none, label is no longer used in this category.
	    	Document suggestDocument = getDocument(FIELD_SUGGEST_LABEL_KEY, label);
	    	String[] suggestDocumentCategories = suggestDocument.getValues(FIELD_SUGGEST_INSTANCE_CATEGORY);
    		// Remove all categories, add back all but the unused one.
    		suggestDocument.removeFields(FIELD_SUGGEST_INSTANCE_CATEGORY);
    		logger.debug("No use of label [" + label + "] in category: " + category);
    		for (String existingCategory : suggestDocumentCategories) {
    			if (existingCategory.equals(category)) {
		    		logger.debug("          category " + existingCategory + "  [removed]");
    				continue;
    			}
	    		logger.debug("          category " + existingCategory);
                suggestDocument.add(new Field(FIELD_SUGGEST_INSTANCE_CATEGORY, existingCategory, Field.Store.YES, Field.Index.NOT_ANALYZED));
    		}
    		if (suggestDocument.getValues(FIELD_SUGGEST_INSTANCE_CATEGORY).length == 0) {
	    		logger.debug("No use of label in any category: " + label);
	    		deleteDocument(FIELD_SUGGEST_LABEL_KEY, label);
	    		// If this label was a class label (or synonym), clean out class entity from index
	    		//         Search entities:  (!isInstance && label == labelReference)
	    		// TODO remove class entity
	    		// TODO assert that there is now no entity with this label in the index
    		}
		}
    }

    //   Search entities:  (isInstance && has label reference == labelReference && has category == category)
	private boolean isLabelReference(String label, String category) {
		// Commit cache so that the search read is up to date
		commitDocumentCache(null);
		try {
	        final BooleanQuery query = new BooleanQuery();
            query.add(new TermQuery(new Term(FIELD_ENTITY_IS_INSTANCE, Boolean.toString(true))), BooleanClause.Occur.MUST);
            query.add(new TermQuery(new Term(FIELD_ENTITY_LABEL_REFERENCE, label)), BooleanClause.Occur.MUST);
            query.add(new TermQuery(new Term(FIELD_ENTITY_INSTANCE_CATEGORY, category)), BooleanClause.Occur.MUST);
	        // TODO shouldn't have to new one up
	        final IndexSearcher searcher = new IndexSearcher(directory, true);
	        final TopDocs docs = searcher.search(query, 1);
    		logger.debug("isLabelReference: label:" + label + "  category:" + category + "  " + (docs.totalHits > 0));
	        return (docs.totalHits > 0);
		} catch (IOException e) {
			logger.error(e);
			return true;
		}
	}
    
}
