package org.eaglei.ui.gwt.search.results;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;

import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.ui.gwt.ApplicationContext;
import org.eaglei.ui.gwt.search.SearchContext;
import org.eaglei.ui.gwt.search.results.SideBar.Component;
import org.eaglei.model.gwt.rpc.ClientModelManager;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.ClassDefinitionCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.SubClassCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.SuperClassesCallback;
import org.eaglei.model.gwt.rpc.ClientModelManager.TopLevelClassesCallback;

import com.google.gwt.event.dom.client.ClickEvent;
import com.google.gwt.event.dom.client.ClickHandler;
import com.google.gwt.event.dom.client.MouseOutEvent;
import com.google.gwt.event.dom.client.MouseOutHandler;
import com.google.gwt.event.dom.client.MouseOverEvent;
import com.google.gwt.event.dom.client.MouseOverHandler;
import com.google.gwt.user.client.DOM;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.UIObject;
import com.google.gwt.user.client.ui.VerticalPanel;

public class ResourceNavigatorComponent extends Component {

    private static class ResourceNavigatorPanel extends Composite {
        private static final int MAX_TOPICS = 5;
        private static final int MARGIN_INCREMENT = 6; // 
        private static final String STANDARD_TEXT_INDENT = "-10px"; //
        private static final String ARROW_TEXT_INDENT = "-14px"; // extra indent for <
        private final VerticalPanel outer = new VerticalPanel();
        private int currentMargin;
        private SearchRequest request;
        private EIURI currentURI = null;
        
        ResourceNavigatorPanel() {
            super();
            initWidget(outer);
        }
        
        void setSearchRequest(SearchRequest request) {
            this.request = request;
            EIURI uriResultType = request.getBinding() != null ? request.getBinding().getType() : null;
            setResource(uriResultType);
        }
        
        private UIObject addLabel(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            UIObject ui = addLabel(c.getEntity().getLabel());
            setClassDefinitionTooltip(c, ui, mapNoDefClassToUIObject);
            return ui;
        }
        
        private UIObject addLabel(String labelString) {
            Label l = new Label(labelString);
            DOM.setStyleAttribute(l.getElement(), "textIndent", STANDARD_TEXT_INDENT);
            DOM.setStyleAttribute(l.getElement(), "marginLeft", currentMargin+"px");
            DOM.setStyleAttribute(l.getElement(), "fontWeight", "bold");
            outer.add(l);
            return l;
        }

        /*
         * Adds a link for an ancestor class of the search request type binding
         */
        private UIObject addAncestorLink(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            UIObject ui = addLink(c, true, false, mapNoDefClassToUIObject);
            return ui;
        }
        
        /*
         * Adds a link for a subclass of the search request type binding
         */
        private UIObject addSubClassLink(EIClass c, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            UIObject ui = addLink(c, false, c.hasSubClass(), mapNoDefClassToUIObject);
            return ui;
        }
        
        /*
         * Adds a link for a class
         */
        private UIObject addLink(EIClass c, boolean displayArrow, boolean bold, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            UIObject ui = addLink(c.getEntity().getLabel(), c.getEntity().getURI(), displayArrow, bold);
            setClassDefinitionTooltip(c, ui, mapNoDefClassToUIObject);
            return ui;
        }
        
        private Label addLink(final String labelString, final EIURI typeBindingURI, boolean displayArrow, boolean bold) {
            final Label l;
            if (displayArrow) {
                l = new Label("< "+labelString);
                DOM.setStyleAttribute(l.getElement(), "textIndent", ARROW_TEXT_INDENT);
            } else {
                l = new Label(labelString);
                DOM.setStyleAttribute(l.getElement(), "textIndent", STANDARD_TEXT_INDENT);
            }
            DOM.setStyleAttribute(l.getElement(), "marginLeft", currentMargin+"px");
            if (bold) {
                DOM.setStyleAttribute(l.getElement(), "fontWeight", "bold");
            }
            l.setStyleName("link");
            l.addMouseOverHandler(new MouseOverHandler() {
                
                @Override
                public void onMouseOver(MouseOverEvent event) {
                    l.addStyleDependentName("hovering");
                }
                
            });
            l.addMouseOutHandler(new MouseOutHandler() {
                
                @Override
                public void onMouseOut(MouseOutEvent event) {
                    l.removeStyleDependentName("hovering");
                }
                
            });
            l.addClickHandler(new ClickHandler() {
                
                @Override
                public void onClick(ClickEvent event) {
                    SearchRequest newRequest = new SearchRequest();
                    newRequest.setInstitution(request.getInstitution());
                    if (request.getTerm() != null) {
                        newRequest.setTerm(new SearchRequest.Term(request.getTerm()));
                    }
                    if (typeBindingURI != null) {
                        newRequest.setBinding(new SearchRequest.TypeBinding(typeBindingURI));
                    }
                    SearchContext.INSTANCE.search(request, false);
                }
            });
            outer.add(l);
            return l;
        }

        private void setResource(EIURI uri) {
            // If component was previously populated from the
            // same URI, no-op
            if (outer.getWidgetCount() > 0) {
                if (uri != null && uri.equals(currentURI)) {
                    return;
                } else if (uri == null && currentURI == null) {
                    return;
                }
            }
            currentURI = uri;
            outer.setVisible(false);
            outer.clear();
            currentMargin = 18;
            if (uri == null) {
                // No type binding, display top-level resources
                addLabel("Any Resource");
                
                ClientModelManager.INSTANCE.getTopLevelClasses(new TopLevelClassesCallback() {

                    @Override
                    public void onSuccess(List<EIClass> result) {
                        currentMargin += MARGIN_INCREMENT;
                        HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                        for (EIClass top : result) {
                            addSubClassLink(top, mapNoDefClassToUIObject);
                        }
                        getClassDefinitions(mapNoDefClassToUIObject);
                        outer.setVisible(true);
                    }
                });
            } else {
                addLink("All Resources", null, true, false);
                ClientModelManager.INSTANCE.getClass(uri, new ClassCallback() {

                    @Override
                    public void onSuccess(EIClass result) {
                        setSuperClasses(result);
                    }

                });
            }
        }

        private void setSuperClasses(EIClass c) {
            if (!c.hasSuperClass()) {
                // Top-level class, no supers
                currentMargin += MARGIN_INCREMENT;
                HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                addLabel(c, mapNoDefClassToUIObject);
                getClassDefinitions(mapNoDefClassToUIObject);
                setSubClasses(c);
            } else {
                ClientModelManager.INSTANCE.getSuperClasses(c, new SuperClassesCallback() {

                    @Override
                    public void onSuccess(EIClass result) {
                        ArrayList<EIClass> listAncestors = new ArrayList<EIClass>();
                        EIClass parent = result.getSuperClass();
                        while (parent != null) {
                            listAncestors.add(parent);
                            parent = parent.getSuperClass();
                        }
                        HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                        for (int i = listAncestors.size() - 1; i >= 0; i--) {
                            // Supers
                            currentMargin += MARGIN_INCREMENT;
                            addAncestorLink(listAncestors.get(i), mapNoDefClassToUIObject);
                        }
                        // The class
                        currentMargin += MARGIN_INCREMENT;
                        addLabel(result, mapNoDefClassToUIObject);
                        getClassDefinitions(mapNoDefClassToUIObject);
                        setSubClasses(result);
                    }

                });
            }
        }

        private void setSubClasses(final EIClass c) {
            if (!c.hasSubClass()) {
                // No subclasses
                outer.setVisible(true);
            } else {
                ClientModelManager.INSTANCE.getSubClasses(c, new SubClassCallback() {

                    @Override
                    public void onSuccess(EIClass result) {
                        // Subclasses
                        currentMargin += MARGIN_INCREMENT;
                        HashMap<EIClass,UIObject> mapNoDefClassToUIObject = new HashMap<EIClass,UIObject>();
                        for (EIClass child : result.getSubClasses()) {
                            addSubClassLink(child, mapNoDefClassToUIObject);
                        }
                        getClassDefinitions(mapNoDefClassToUIObject);
                        outer.setVisible(true);
                    }

                });
            }
        }
        
        /*
         * 
         */
        private void setClassDefinitionTooltip(EIClass clazz, UIObject ui, HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            // Set tooltip with the definition.
            // If definition is not in cache, it will be fetched and
            // set async.
            if (clazz.getDefinition() == null) {
                mapNoDefClassToUIObject.put(clazz, ui);
            } else {
                ui.setTitle(clazz.getDefinition());
            }
        }

        private void getClassDefinitions(final HashMap<EIClass,UIObject> mapNoDefClassToUIObject) {
            // Asynchronously populate tooltips
            if (mapNoDefClassToUIObject.size() > 0) {
                final List<EIClass> listNoDefClasses = new ArrayList<EIClass>(mapNoDefClassToUIObject.keySet());
                ClientModelManager.INSTANCE.getClassDefinitions(listNoDefClasses, new ClassDefinitionCallback() {
                    @Override
                    public void onSuccess(List<EIClass> result) {
                        for (EIClass clazz : result) {
                            UIObject o = mapNoDefClassToUIObject.get(clazz);
                            o.setTitle(clazz.getDefinition());
                        }
                    }
                });
            }            
        }
    }

    private ResourceNavigatorPanel p;

    public ResourceNavigatorComponent() {
        super("Resource");
        p = new ResourceNavigatorPanel();
        setWidget(p);
    }

    public void setSearchRequest(SearchRequest request) {
        p.setSearchRequest(request);
    }
}
