package org.eaglei.ui.gwt.search;

import java.util.ArrayList;

import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.ui.gwt.ApplicationContext;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemote;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemoteAsync;
import org.eaglei.ui.gwt.security.SessionContext;
import org.eaglei.ui.gwt.security.SessionContext.SessionListener;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.Window;
import com.google.gwt.user.client.rpc.AsyncCallback;

public class SearchContext implements SessionListener {
	
	public interface SearchListener {
		void onRequest(SearchRequest request, boolean isTypeCategoryChange, boolean isPageRequest);
		void onResults(ClientSearchResultSet results);
		// TODO error code
		void onFailure(SearchRequest request, String message);
	}
	
    public static final SearchServiceRemoteAsync searchService = GWT.create(SearchServiceRemote.class);
	public static final SearchContext INSTANCE = new SearchContext();
	
	private boolean pending = false;
    private SearchRequest currentRequest = null;
    private ClientSearchResultSet currentResults = null;
	private ArrayList<SearchListener> listeners = new ArrayList<SearchListener>();
	
	private SearchContext() {
		SessionContext.INSTANCE.addListener(this);
	}
	
	public void addListener(SearchListener listener) {
		listeners.add(listener);
	}
	
	public boolean isPending() {
		return pending;
	}
	
    public SearchRequest getCurrentRequest() {
        return currentRequest;
    }
    
    public ClientSearchResultSet getCurrentResults() {
        return currentResults;
    }
    
    /**
     * Sets a request in the context, but doesn't invoke a search.
     * 
     * @param request
     */
    public void setCurrentRequest(final SearchRequest request) {
        if (isNewRequest(request)) {
            setCurrentRequestInternal(request, true, false);
        }
    }
    
    private void setCurrentRequestInternal(final SearchRequest request, boolean isTypeCategoryChange, boolean isPageRequest) {
        this.currentRequest = request;
        this.currentResults = null;
        for (SearchListener listener : listeners) {
            listener.onRequest(request, isTypeCategoryChange, isPageRequest);
        }
    }
    
    private boolean isNewRequest(final SearchRequest request) {
        if (currentRequest == null && request != null) {
            return true;
        }
        return ! currentRequest.equals(request);
    }
    
    private void errorNotification(SearchRequest request, String message) {
        for (SearchListener listener : listeners) {
            listener.onFailure(request, message);
        }        
    }
    
    /**
     * Clears any current or pending search request.
     */
	public void clearSearch() {
		this.pending = false;
		setCurrentRequestInternal(null, false, false);
	}
	
    /**
     * Executes a search if request is non-null.
     */
    public void search(final SearchRequest request) {
        search(request, true, false);
    }
    
    /**
     * Executes a search if request is non-null.
     * 
     * isTypeCategoryChange indicates to search listeners that the 
     * pending search request probably does not involve a change
     * in the type category of the TypeBinding in the search request.
     * This allows certain listeners to avoid updating type category controls
     * while the search is pending.  
     * 
     * The value of this parameter is advisory only. If set to false,
     * and the search ultimately does involve a type category change, 
     * there should be no harm.  The real type category will be
     * indicated in the ClientSearchResultSet, and controls should
     * respect that setting when the search result is received.
     * 
     * isPageRequest indicates that the only difference between
     * this request and the previously executed request is 
     * a change in the start index.
     * 
     * The isPageRequest param is a performance optimization only,
     * no validation is performed.  Do not set it to true
     * unless you are absolutely sure that the previous request
     * is the same as this one except for start index. Setting it to false 
     * when the request is in fact a pagination operation
     * should cause no harm.
     * 
     * @param request
     */
    public void search(final SearchRequest request, boolean isTypeCategoryChange, boolean isPageRequest) {
        if (SessionContext.getSessionId() == null) {
            errorNotification(request, "Please login");
            return;
        }
	    if (! isNewRequest(request)) {
	        return;
	    }
	    if (request == null) {
	        clearSearch();
	    } else {
	        if (currentRequest == null) {
	            // This could be an assert...
	            isPageRequest = false;
	        }
	        // A new, non-null request.  Execute a search.
	        this.pending = true;
	        setCurrentRequestInternal(request, isTypeCategoryChange, isPageRequest);
	        searchService.search(SessionContext.getSessionId(), request, new AsyncCallback<ClientSearchResultSet>() {

	            public void onFailure(Throwable caught) {
	                if (SearchContext.this.currentRequest == null 
	                        || ! SearchContext.this.currentRequest.equals(request)) {
	                    // Failure of an old request. no-op.
	                    return;
	                }
                    SearchContext.this.pending = false;
                    SearchContext.this.currentRequest = null;
	                // Notify listener of failure
	                errorNotification(request, caught.getMessage());
	            }

	            public void onSuccess(ClientSearchResultSet results) {
	                if (SearchContext.this.currentRequest == null 
	                        || ! SearchContext.this.currentRequest.equals(results.getResultSet().getRequest())) {
	                    // Response from an old request. no-op.
	                    return;
	                }
	                // TODO do we need to ensure that any instances of EIClass in
	                // the result data structure are converted to instances in 
	                // the cache?
	                pending = false;
	                currentResults = results;
	                for (SearchListener listener : listeners) {
	                    listener.onResults(results);
	                }
	            }
	            
	        });
            ApplicationContext.setHistory(ApplicationContext.RESULTS_PAGE_TOKEN, request.toURLParams());
	    } 
	}

    @Override
    public void onLogIn() {
        
    }

    @Override
    public void onLogOut() {
        // Clear the context on log out
        clearSearch();
    }

}
