package org.eaglei.ui.gwt.security.server;

import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;

public class SessionManager {

    private final static HashMap<String, Long> mapSessionIdToLastUse = new HashMap<String, Long>();
    private static final Log logger = LogFactory.getLog(SessionManager.class.getName());
    // system property uname and password may be null, but
    // if they aren't, use them to authenticate
    private static final String SYSTEM_USERNAME = System.getProperty("search.username");
    private static final String SYSTEM_PASSWORD = System.getProperty("search.password");

    public final static long EXPIRE_TIME = 1800000;

    private static boolean allowHttpLogin() {
        String allowHttpLogin = System.getProperty("allow.http.login");
        return (allowHttpLogin != null && Boolean.parseBoolean(allowHttpLogin));
        // return true;
    }

    public static void checkHttps(HttpServletRequest request) throws InvalidSessionIdException {
        if (!"https".equals(request.getScheme())) {
            if (!allowHttpLogin()) {
                throw new InvalidSessionIdException("https required");
            }
        }
    }
    
    public static void authenticate(String username, String password) throws InvalidSessionIdException {
        if (SYSTEM_USERNAME != null && !SYSTEM_USERNAME.equals(username)) {
            throw new InvalidSessionIdException("Invalid username/password.");
        }

        if (SYSTEM_PASSWORD != null && !SYSTEM_PASSWORD.equals(password)) {
            throw new InvalidSessionIdException("Invalid username/password.");
        }
    }

    public static String addSession(HttpServletResponse response) {
        // Instead of continually running daemon thread that cleans out sessions,
        // just clean the cache when someone new is logging in.
        removeExpiredSessions();
        
        SecureRandom random = new SecureRandom();
        byte bytes[] = new byte[20];
        random.nextBytes(bytes);

        String sessionId = String.valueOf(bytes);

        synchronized (SessionManager.class) {
            mapSessionIdToLastUse.put(sessionId, System.currentTimeMillis());
        }
        logger.info("addSession: " + sessionId);

        Cookie c = new Cookie("SearchLogSID", sessionId);
        if (!allowHttpLogin()) {
            c.setSecure(true);
        }
        response.addCookie(c);

        return sessionId;
    }

    public static void removeSession(HttpServletRequest request) {
        try {
            String sessionId = getSessionIdFromCookie(request);
            removeSession(sessionId);
        } catch (InvalidSessionIdException e) {
        }
    }

    public static synchronized void removeSession(String sessionId) {
        if (sessionId != null) {
            mapSessionIdToLastUse.remove(sessionId);
            logger.info("removed session: " + sessionId);
        }
    }

    /**
     * Returns the set of expired session ids.
     * 
     * @return
     */
    public static synchronized List<String> removeExpiredSessions() {
        ArrayList<String> invalid = new ArrayList<String>();
        for (String sessionId : mapSessionIdToLastUse.keySet()) {
            Long last = mapSessionIdToLastUse.get(sessionId);
            // Expire after 30 minutes
            if (last + EXPIRE_TIME < System.currentTimeMillis()) {
                invalid.add(sessionId);
            }
        }
        // Need to remove the session ids inside this sync block
        for (String sessionId : invalid) {
            removeSession(sessionId);
        }
        return invalid;
    }

    public static void validate(HttpServletRequest request) throws InvalidSessionIdException {
        // Currently requiring data requests as well login to be https
        checkHttps(request);
        String sessionId = getSessionIdFromCookie(request);
        validate(sessionId);
    }

    public static synchronized void validate(String sessionId) throws InvalidSessionIdException {
        if (sessionId == null) {
            if (SYSTEM_USERNAME != null || SYSTEM_PASSWORD != null) {
                throw new InvalidSessionIdException("Session id is required");
            }
            // A null session id may be accepted if the system uname and pword are null.
            return;
        }
        Long last = mapSessionIdToLastUse.get(sessionId);
        if (last == null) {
            throw new InvalidSessionIdException("Session expired");
        }
        mapSessionIdToLastUse.put(sessionId, System.currentTimeMillis());
    }

    private static String getSessionIdFromCookie(HttpServletRequest request)
            throws InvalidSessionIdException {
        Cookie[] cookies = request.getCookies();
        for (Cookie c : cookies) {
            if ("SearchLogSID".equals(c.getName())) {
                return c.getValue();
            }
        }
        throw new InvalidSessionIdException("Could not authenticate");
    }
}
