package org.eaglei.ui.gwt.search.server;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;

import javax.servlet.ServletException;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.eaglei.model.EIClass;
import org.eaglei.model.EIEntity;
import org.eaglei.model.EIOntModel;
import org.eaglei.model.EIURI;
import org.eaglei.search.provider.SearchCountRequest;
import org.eaglei.search.provider.SearchCounts;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchResultSet;
import org.eaglei.search.provider.SearchProvider;
import org.eaglei.security.SecurityProvider;
import org.eaglei.security.Session;
import org.eaglei.services.InstitutionRegistry;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;
import org.eaglei.ui.gwt.rpc.LoggedException;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemote;
import org.springframework.context.ApplicationContext;
import org.springframework.context.annotation.AnnotationConfigApplicationContext;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.support.WebApplicationContextUtils;

import com.google.gwt.user.server.rpc.RemoteServiceServlet;

/**
 * Implementation of SearchService.
 */
public class SearchServlet extends RemoteServiceServlet implements SearchServiceRemote {

    private static final Log logger = LogFactory.getLog(SearchServlet.class);
    private static final boolean DEBUG = logger.isDebugEnabled(); 
    
    /*
     * Provider supports execution of search requests (through network or directly against local repo)
     */
    private SearchProvider searchProvider;
    private SecurityProvider securityProvider;
    private EIOntModel eagleiOntModel;
    private InstitutionRegistry institutionRegistry;
    private List<EIClass> searchCategoryClasses;
    private HashSet<EIURI> searchCategoryURIs;

    public SearchServlet() {
    }
    
    @Override
    public void init() {
        WebApplicationContext ctx = 
            WebApplicationContextUtils.getWebApplicationContext(getServletContext());
        searchProvider = ctx.getBean("rootSearchProvider", SearchProvider.class);
        securityProvider = ctx.getBean("securityProvider", SecurityProvider.class);
        eagleiOntModel = ctx.getBean(EIOntModel.class);
        institutionRegistry = ctx.getBean(InstitutionRegistry.class);
        List<String> searchCategories = (List<String>) ctx.getBean("searchCategories");
        searchCategoryClasses = new ArrayList<EIClass>(searchCategories.size());
        searchCategoryURIs = new HashSet<EIURI>(searchCategories.size());
        for (String uriStr : searchCategories) {
            EIURI uri = EIURI.create(uriStr);
            searchCategoryClasses.add(eagleiOntModel.getClass(uri));
            searchCategoryURIs.add(uri);
        }
        try {
            searchProvider.init();
            logger.info("SearchProvider created and initialized");
        } catch (Throwable t) {
            logger.error("Initialization of search provider failed", t);
            throw new RuntimeException("Initialization of search provider failed", t);
        }
    }

    @Override
    public void destroy() {
        //TODO does Spring need to be gracefully shutdown?
    }

    /* (non-Javadoc)
     * @see org.eaglei.search.client.rpc.SearchService#getInstitutions(java.lang.String)
     */
    public List<EIEntity> getInstitutions() throws LoggedException, InvalidSessionIdException {
        //SessionManager.validate(sessionId);
        try {
           
            return institutionRegistry.getInstitutions();
           
        } catch (Throwable t) {
            logger.error("Error in getInstitutions", t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }
    
    public List<EIClass> getTopLevelSearchCategories() throws LoggedException {
        //SessionManager.validate(sessionId);
        try {
           
            return searchCategoryClasses;
           
        } catch (Throwable t) {
            logger.error("Error in getTopLevelSearchCategories", t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

    
    /* (non-Javadoc)
     * @see org.eaglei.search.client.rpc.SearchService#getSearchResults(java.lang.String, org.eaglei.search.request.SearchRequest)
     */
    public ClientSearchResultSet search(final String sessionId, final SearchRequest request) throws InvalidSessionIdException, LoggedException {
        Session session = securityProvider.getSession(sessionId);
        if (session == null) {
            throw new InvalidSessionIdException("Invalid session id.");
        }
        try {
            // TODO session id errors must be propagated
            // to the client as a checked exception or error
            // code in the return object.
            if (request == null) {
                logger.error("Null search request");
                throw new LoggedException();
            }
            if (DEBUG) {
                logger.debug(request.toString());
            }

            // execute the search
            SearchResultSet resultSet = searchProvider.query(request);
            
            // Lookup the entity for the type binding.
            EIEntity bindingEntity =
                (request.getBinding() != null) ? eagleiOntModel.getClass(request.getBinding().getType()).getEntity() : null;
            
            // Compute the tab category of the binding
            EIURI bindingCategoryURI = null;
            if (request.getBinding() != null) {
                if (searchCategoryURIs.contains(request.getBinding().getType())) {
                    bindingCategoryURI = request.getBinding().getType();
                } else {
                    List<EIClass> superclasses = eagleiOntModel.getSuperClasses(request.getBinding().getType());
                    for (int i=superclasses.size()-1; i>=0; i--) {
                        EIURI superclassURI = superclasses.get(i).getEntity().getURI();
                        if (searchCategoryURIs.contains(superclassURI)) {
                            bindingCategoryURI = superclassURI;
                            break;
                        }
                    }
                }
            }
            
            // Generate a lookup table for the root class of each search result type
            HashMap<EIURI, EIEntity> mapURIToRootEntity = new HashMap<EIURI, EIEntity>();
            for (SearchResult result : resultSet.getResults()) {
                List<EIClass> superclasses = eagleiOntModel.getSuperClasses(result.getType().getURI());
                if (superclasses.size() > 0) {
                    mapURIToRootEntity.put(result.getType().getURI(), superclasses.get(superclasses.size()-1).getEntity());
                }
            }

            return new ClientSearchResultSet(resultSet, bindingEntity, bindingCategoryURI, mapURIToRootEntity);
           
        } catch (Throwable t) {
            logger.error("Unexpected error in search: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }
    }
    
    public SearchCounts count(final String sessionId, SearchCountRequest request) throws InvalidSessionIdException, LoggedException {
        Session session = securityProvider.getSession(sessionId);
        if (session == null) {
            throw new InvalidSessionIdException("Invalid session id.");
        }
        try {
            // TODO session id errors must be propagated
            // to the client as a checked exception or error
            // code in the return object.
            if (request == null) {
                logger.error("Null search request");
                throw new LoggedException();
            }
            if (DEBUG) {
                logger.debug(request.toString());
            }

            // Implementing server-side for a little perf benefit.
            for (EIClass c : searchCategoryClasses) {
                EIURI uri = c.getEntity().getURI();
                if (request.getRequest().getBinding() == null 
                        || ! uri.equals(request.getRequest().getBinding().getType())) {
                    request.addCountType(uri);
                }
            }
            if (request.getRequest().getBinding() != null) {
                request.addCountType(null);  // All results
            }
            
            // execute the search
            return searchProvider.count(request);                
           
        } catch (Throwable t) {
            logger.error("Unexpected error in count: " + request.toString(), t);
            throw new LoggedException(t.getLocalizedMessage());
        }        
    }

}
