package org.eaglei.ui.gwt.search;

import java.util.ArrayList;

import org.eaglei.search.provider.SearchRequest;
import org.eaglei.ui.gwt.ApplicationContext;
import org.eaglei.ui.gwt.rpc.InvalidSessionIdException;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemote;
import org.eaglei.ui.gwt.search.rpc.SearchServiceRemoteAsync;
import org.eaglei.ui.gwt.security.SessionContext;
import org.eaglei.ui.gwt.security.SessionContext.SessionListener;

import com.google.gwt.core.client.GWT;
import com.google.gwt.user.client.rpc.AsyncCallback;

public class SearchContext implements SessionListener {
	
	public interface SearchListener {
	    /**
	     * A new search request is about to be executed.  The request has
	     * not been added to the history stack.  Components can
	     * write their state into the request as is appropriate.
	     * 
	     * @param request
	     */
        void onRequestCreate(SearchRequest request);
        /**
         * The given request is in the history stack and the
         * search operation is pending.
         * Listeners should not modify the request.
         * 
         * @param request
         * @param isPageRequest
         */
        void onRequestPending(SearchRequest request, boolean isPageRequest);
		void onResults(ClientSearchResultSet results);
		// TODO error code
		void onFailure(SearchRequest request, String message);
	}
	
    public static final SearchServiceRemoteAsync searchService = GWT.create(SearchServiceRemote.class);
	public static final SearchContext INSTANCE = new SearchContext();
	
    private SearchRequest currentRequest = null;
    private ClientSearchResultSet currentResults = null;
	private ArrayList<SearchListener> listeners = new ArrayList<SearchListener>();
	
	private SearchContext() {
		SessionContext.INSTANCE.addListener(this);
	}
	
	public void addListener(SearchListener listener) {
		listeners.add(listener);
	}
	
    public SearchRequest getCurrentRequest() {
        return currentRequest;
    }
    
    public ClientSearchResultSet getCurrentResults() {
        return currentResults;
    }
    
    /**
     * Sets a request in the context, but doesn't invoke a search.
     * 
     * @param request
     */
    public void setFromHistoryParams(final String params) {
        final SearchRequest request = new SearchRequest(params);
        if (!isNewRequest(request)) {
            // What about a Refresh?
            return;
        }
        SearchRequest previous = this.currentRequest;
        this.currentRequest = request;
        this.currentResults = null;
        boolean isPageRequest = request != null ? request.equals(previous, true) : false;
        for (SearchListener listener : listeners) {
            listener.onRequestPending(request, isPageRequest);
        }
        searchService.search(SessionContext.getSessionId(), request, new AsyncCallback<ClientSearchResultSet>() {

            public void onFailure(Throwable caught) {
                if (caught instanceof InvalidSessionIdException) {
                    SessionContext.INSTANCE.logOut();
                    return;
                }
                if (SearchContext.this.currentRequest == null 
                        || ! SearchContext.this.currentRequest.equals(request)) {
                    // Failure of an old request. no-op.
                    return;
                }
                SearchContext.this.currentRequest = null;
                // Notify listener of failure
                errorNotification(request, caught.getMessage());
            }

            public void onSuccess(ClientSearchResultSet results) {
                if (SearchContext.this.currentRequest == null 
                        || ! SearchContext.this.currentRequest.equals(results.getResultSet().getRequest())) {
                    // Response from an old request. no-op.
                    return;
                }
                // TODO do we need to ensure that any instances of EIClass in
                // the result data structure are converted to instances in 
                // the cache?
                currentResults = results;
                for (SearchListener listener : listeners) {
                    listener.onResults(results);
                }
            }
            
        });
    }
    
    private boolean isNewRequest(final SearchRequest request) {
        if (currentRequest == null && request != null) {
            return true;
        }
        return ! currentRequest.equals(request);
    }
    
    private void errorNotification(SearchRequest request, String message) {
        for (SearchListener listener : listeners) {
            listener.onFailure(request, message);
        }        
    }
    
    /**
     * 
     * 
     * @param request
     */
    public void search(SearchRequest request) {
        if (SessionContext.getSessionId() == null) {
            onLogOut();
            return;
        }
	    if (request == null) {
	        // or assert?
            request = new SearchRequest();
	    }
        for (SearchListener listener : listeners) {
            listener.onRequestCreate(request);
        }
        ApplicationContext.setHistory(ApplicationContext.RESULTS_PAGE_TOKEN, request.toURLParams());
	}

    @Override
    public void onLogIn() {
        
    }

    @Override
    public void onLogOut() {
        errorNotification(this.currentRequest, "Please login");
        this.currentRequest = null;
        this.currentResults = null;
    }

}
