package org.eaglei.ui.gwt.search.results;

import java.util.HashMap;
import java.util.Map;

import org.eaglei.model.EIEntity;
import org.eaglei.search.provider.SearchRequest;
import org.eaglei.search.provider.SearchResult;
import org.eaglei.search.provider.SearchRequest.DataSet;
import org.eaglei.ui.gwt.ApplicationContext;
import org.eaglei.ui.gwt.ApplicationResources;
import org.eaglei.ui.gwt.SearchApplicationContext;
import org.eaglei.ui.gwt.search.SearchContext;
import org.eaglei.ui.gwt.search.rpc.ClientSearchResultSet;

import com.google.gwt.user.client.ui.Anchor;
import com.google.gwt.user.client.ui.FlexTable;
import com.google.gwt.user.client.ui.HTML;
import com.google.gwt.user.client.ui.Hyperlink;
import com.google.gwt.user.client.ui.Image;
import com.google.gwt.user.client.ui.Label;
import com.google.gwt.user.client.ui.VerticalPanel;
import com.google.gwt.user.client.ui.Widget;

public class ResourcesGrid extends FlexTable implements NavBar.NavListener {
    
    public static class CellPanel extends VerticalPanel {
        
        CellPanel() {
            setStyleName("resultsTableCell");
            setSpacing(0);
        }
        
        void setContent(Widget primary, Widget secondary) {
            clear();
            add(primary);
            primary.addStyleName("primary");
            if (secondary != null) {
                add(secondary);
                secondary.setStyleName("secondary");
            }
        }
    }
    
    public static class DataSetRenderer {
        
        protected String[] columnHeaderNames;
        protected DataSet dataset;
        protected ResourcesGrid resultsTable;
        
        public DataSetRenderer(ResourcesGrid resultsTable, DataSet dataset, String[] columnHeaderNames) {
            this.resultsTable = resultsTable;
            this.dataset = dataset;
            this.columnHeaderNames = columnHeaderNames;
        }
        
        public DataSet getDataset() {
            return this.dataset;
        }
        
        public String[] getHeaderColumnNames() {
            return columnHeaderNames;
        }
        
        public void displayHeaderRow() {
            int col = 0;
            for (String colName : getHeaderColumnNames()) {
                resultsTable.getCellFormatter().setStyleName(HEADER_ROW, col, "listGridHeader");
                resultsTable.setHTML(HEADER_ROW, col++, "<b>" + colName + "</b>");
            }
        }
        
        public void createInstanceRow(int row, DataSet dataset, SearchResult searchResult) {
            int col = 0;
            
            // Name cell
            Widget instanceLink;
            if (dataset == DataSet.EI_RESOURCES) {
                instanceLink = new Hyperlink(searchResult.getEntity().getLabel(), 
                        SearchApplicationContext.INSTANCE_PAGE_TOKEN + "uri=" + searchResult.getEntity().getURI().toString());
            } else {
                instanceLink = new Anchor(searchResult.getEntity().getLabel(), searchResult.getEntity().getURI().toString());
                ((Anchor) instanceLink).setTarget("_blank"); // display in new tab
            }
            HTML snippit = null;
            if (searchResult.getHighlight() != null) {
                snippit = new HTML(searchResult.getHighlight());
            }
            CellPanel cp = new CellPanel();
            cp.setContent(instanceLink, snippit);
            resultsTable.setWidget(row, col++, cp);

            // Type cell
            Label typeLabel = new Label(searchResult.getType().getLabel());
            Label rootTypeLabel = null;
            EIEntity rootTypeEntity = resultsTable.currentResults.getMapURIToRootEntity().get(searchResult.getType().getURI());
            if (rootTypeEntity != null) {
                rootTypeLabel = new Label(rootTypeEntity.getLabel());
            }
            cp = new CellPanel();
            cp.setContent(typeLabel, rootTypeLabel);
            if (rootTypeLabel != null) {
                rootTypeLabel.addStyleName("rootType");
            }
            resultsTable.setWidget(row, col++, cp);
            
            // Location cell
            EIEntity lab = searchResult.getLab();
            Hyperlink labLink;
            if (lab != null) {
                labLink = new Hyperlink(lab.getLabel(), 
                    SearchApplicationContext.INSTANCE_PAGE_TOKEN + "uri=" + lab.getURI().toString());
            } else {
                labLink = null;
            }
            Label instLabel = new Label(searchResult.getInstitution().getLabel());
            cp = new CellPanel();
            if (ApplicationContext.IS_INSTITUTION) {
                if (labLink != null) {
                    // If in institution mode, display Lab as the primary label
                    cp.setContent(labLink, null);
                }
            } else {
                // If in Central mode, display Institution as primary, lab as secondary.
                cp.setContent(instLabel, labLink);
            }
            resultsTable.setWidget(row, col++, cp);
        }

    }
    
    private static final int PAGE_SIZE = SearchRequest.DEFAULT_PAGE_SIZE;
    //private static final int NAV_BAR_ROW = 0;
    private static final int HEADER_ROW = 0;
    private static final int STATUS_ROW = 1;
    private static final int PROGRESS_ROW = 2;
    private static final int FIRST_RESULT_ROW = 3;
    
	private final Map<DataSet, DataSetRenderer> mapDSToRenderer = new HashMap<DataSet, ResourcesGrid.DataSetRenderer>();;
	private ClientSearchResultSet currentResults;
	private DataSetRenderer currentRenderer;
	private final NavBar navBar;
	private Label status;
	
	public ResourcesGrid(NavBar navBar) {
        // Header columns
        setStyleName("listGrid");
        this.navBar = navBar;
        navBar.addNavListnener(this);
        status = new Label();
        setWidget(STATUS_ROW, 0, status);
        setWidget(PROGRESS_ROW, 0, new Image(ApplicationResources.INSTANCE.loading()));
	}
	
	public ClientSearchResultSet getCurrentResults() {
	    return this.currentResults;
	}
	
	public void addRenderer(DataSetRenderer renderer) {
        this.mapDSToRenderer.put(renderer.getDataset(), renderer);
	}
	
	public void displaySearchPending() {
        currentResults = null;
        RowFormatter rf = getRowFormatter();
        navBar.setVisible(false);
        rf.setVisible(STATUS_ROW, false);
        rf.setVisible(PROGRESS_ROW, true);
	    for (int i = getRowCount()-1; i>=FIRST_RESULT_ROW; i--) {
	        removeRow(i);
	    }
	}
	
	public void displayStatusMessage(String message) {
        status.setText(message);
        currentResults = null;
        RowFormatter rf = getRowFormatter();
        navBar.setVisible(false);
        rf.setVisible(STATUS_ROW, true);
        rf.setVisible(PROGRESS_ROW, false);
        for (int i = getRowCount()-1; i>=FIRST_RESULT_ROW; i--) {
            removeRow(i);
        }
	}
	
	public void displaySearchResults(ClientSearchResultSet results) {
        if (results.getResultSet().getResults().size() == 0) {
            displayStatusMessage("No results found.");
            return;
        }
        
        RowFormatter rf = getRowFormatter();
        
        if (currentRenderer == null 
                || currentRenderer.getDataset() != results.getResultSet().getRequest().getDataset()) {
            currentRenderer = mapDSToRenderer.get(results.getResultSet().getRequest().getDataset());
            currentRenderer.displayHeaderRow();
        }
        
	    currentResults = results;
        rf.setVisible(PROGRESS_ROW, false);

        // Update the nav bar.
        int count = results.getResultSet().getTotalCount();
        int max = results.getResultSet().getStartIndex() + PAGE_SIZE;
        if (max > count) {
            max = count;
        }
        navBar.update(results.getResultSet().getStartIndex(), count, max);
        navBar.setVisible(true);
        
        // Add the results
        rf.setVisible(STATUS_ROW, false);
        int row = FIRST_RESULT_ROW;
        DataSet dataset = results.getResultSet().getRequest().getDataset();
        for (SearchResult result : results.getResultSet().getResults()) {
            currentRenderer.createInstanceRow(row++, dataset, result);
        }
        
	}
	
    @Override
    public void onPreviousPage() {
        // Move back a page.
        int startIndex = currentResults.getResultSet().getStartIndex() - PAGE_SIZE;
        assert(startIndex >= 0) : "paged beyond the start of the result list";
        executeNewPageSearch(startIndex);
    }

    @Override
    public void onNextPage() {
        // Move forward a page.
        int startIndex = currentResults.getResultSet().getStartIndex() + PAGE_SIZE;
        assert(startIndex < currentResults.getResultSet().getTotalCount()) : "paged beyond the end of the result list";
        executeNewPageSearch(startIndex);
    }

    private void executeNewPageSearch(int startIndex) {
        SearchRequest newRequest = new SearchRequest();
        SearchRequest oldRequest = currentResults.getResultSet().getRequest();
        newRequest.setInstitution(oldRequest.getInstitution());
        // Is it risky to not create a new term and binding?
        newRequest.setTerm(oldRequest.getTerm());
        newRequest.setBinding(oldRequest.getBinding());
        newRequest.setStartIndex(startIndex);
        SearchContext.INSTANCE.search(newRequest);
    }

}
