package org.eaglei.ui.gwt.suggest;


import java.util.List;

import com.google.gwt.event.dom.client.FocusEvent;
import com.google.gwt.event.dom.client.FocusHandler;
import com.google.gwt.event.dom.client.KeyCodes;
import com.google.gwt.event.dom.client.KeyUpEvent;
import com.google.gwt.event.dom.client.KeyUpHandler;
import com.google.gwt.event.logical.shared.SelectionEvent;
import com.google.gwt.event.logical.shared.SelectionHandler;
import com.google.gwt.user.client.Command;
import com.google.gwt.user.client.DeferredCommand;
import com.google.gwt.user.client.ui.Composite;
import com.google.gwt.user.client.ui.SuggestBox;
import com.google.gwt.user.client.ui.SuggestOracle;
import com.google.gwt.user.client.ui.SuggestOracle.Suggestion;

public abstract class AbstractSearchBox extends Composite {
    
    /**
     * Add a search handler which will be called when a search
     * should be executed.
     */
    public interface SearchHandler {
        void performSearch(String query, String uri);
    }
    
    /*
     * This SearchCommand was added as a way to aggregate the events generated
     * by hitting ENTER when the suggest list is open. In this scenario two
     * event listeners are fired, one for a pick from the suggest list, one a
     * keyboard event from the text box. We attempt to give the suggest list
     * pick precedence. If the suggest list is not open, then the text in the
     * text box gets used.
     * 
     * Hitting the Search button will always use the text in the text box.
     */
    class SearchCommand implements Command {
        private String textQuery;
        private SearchSuggestion textSuggestion;
        private SearchSuggestion listSuggestion;

        public void execute() {
            if (listSuggestion != null) {
                //System.out.println("Seach Command.querySuggest "+querySuggest+"  :  "+queryText);
                handleSearchSelected(listSuggestion.getReplacementString(), listSuggestion.getURIString());
            } else if (textSuggestion != null) {
                //System.out.println("Seach Command.queryText "+queryText+"  :  "+querySuggest);
                handleSearchSelected(textQuery, textSuggestion.getURIString());
            } else if (textQuery != null) {
                handleSearchSelected(textQuery, null);
            }
            textQuery = null;
            textSuggestion = null;
            listSuggestion = null;
        }

        public void setTextQuery(String textQuery, SearchSuggestion textSuggestion) {
            this.textQuery = textQuery;
            this.textSuggestion = textSuggestion;
        }

        public void setSuggestQuery(SearchSuggestion listChoice) {
            this.listSuggestion = listChoice;
        }

        /*
         * 
         */
        private void handleSearchSelected(String query, String uri) {
            if (uri == null && (query == null || query.length() == 0)) {
                //return; // no-op when empty
            }
            handler.performSearch(query, uri);
        }

    }
    
    private SearchHandler handler = null;
    private SearchCommand searchCommand;
  
    protected final SuggestBox sbSearch;

    public AbstractSearchBox(String suggestBaseURL, String style) {
        this(new SearchSuggestOracle(suggestBaseURL), style);
    }
    
    
    public AbstractSearchBox(final SuggestOracle oracle, String style) {
        sbSearch = new SuggestBox(oracle);
        if (style != null) {
            sbSearch.setStyleName(style);
        }
        sbSearch.setAutoSelectEnabled(false);
        sbSearch.setPopupStyleName("SearchBoxPopup");
        sbSearch.getTextBox().addFocusHandler(new FocusHandler() {
            public void onFocus(FocusEvent event) {
                if (getDefaultText().equals(sbSearch.getText())) {
                    setText("");
                }
            }
        });
        
        initWidget(sbSearch);
    }
    
    public SearchHandler getSearchHandler() {
        return this.handler;
    }
    
    public void addSelectionHandler(SelectionHandler<Suggestion> handler) {
    	sbSearch.addSelectionHandler(handler);
    }
    
    public void addSearchHandler(SearchHandler handler, boolean notifyOnListSelect, boolean notifyOnEnter) {
        this.handler = handler;
        if (handler == null) {
            searchCommand = null;
            // TODO remove listeners
            return;
        }
        
        searchCommand = new SearchCommand();
  
        if (notifyOnListSelect) {
            // Search when resource is picked from suggest list
            sbSearch.addSelectionHandler(new SelectionHandler<Suggestion>() {
                public void onSelection(SelectionEvent<Suggestion> event) {
                    SearchSuggestion suggestion = (SearchSuggestion) event.getSelectedItem();
                    searchCommand.setSuggestQuery(suggestion);
                    DeferredCommand.addCommand(searchCommand);
                }
            });
        }

        if (notifyOnEnter) {
            // Search when enter key is pressed
            // Workaround http://code.google.com/p/google-web-toolkit/issues/detail?id=3533
            // Must add handler to text box instead of the SuggestBox composite.
            sbSearch.getTextBox().addKeyUpHandler(new KeyUpHandler() {
                public void onKeyUp(KeyUpEvent event) {
                    if (event.getNativeKeyCode() == KeyCodes.KEY_ENTER) {
                        String query = sbSearch.getText().trim();
                        SearchSuggestion suggestMatch = getSuggestMatch();
                        searchCommand.setTextQuery(query, suggestMatch);
                        DeferredCommand.addCommand(searchCommand);
                    }
                }
            });
        }
    }
    
    /**
     * Checks whether the current text matches an item
     * in the suggest list.
     * 
     * @return SearchSuggestion or null
     */
    protected SearchSuggestion getSuggestMatch() {
        String query = sbSearch.getText().trim();
        List<SearchSuggestion> listSuggestions = 
            ((SearchSuggestOracle) sbSearch.getSuggestOracle()).getCurrentSuggestionList();
        if (listSuggestions != null) {
            String queryLower = query.toLowerCase();
            for (SearchSuggestion suggest : listSuggestions) {
                String suggestLower = suggest.getReplacementString().toLowerCase();
                if (suggestLower.equals(queryLower)) {
                    return suggest;
                }
            }
        }
        return null;
    }
    
    public abstract String getDefaultText();

    public abstract String getSearchBaseURL();
    
    /**
     * Trims leading/trailing whitespace.
     * Returns an empty string if the current text is the default string.
     * 
     * @return
     */
    public String getText() {
        String query = sbSearch.getText().trim();
        if (getDefaultText().equals(query)) {
            return "";
        } else {
            return query;
        }
    }
    
    public void setDefaultText() {
        sbSearch.setText(getDefaultText());
        sbSearch.addStyleDependentName("default");
    }
    
    public void setText(String text) {
        sbSearch.setText(text);
        sbSearch.removeStyleDependentName("default");
    }
    
    /*
    public void setStyleName(String style) {
        sbSearch.setStyleName(style);
        // If the default text is currently being displayed,
        // ensure that the default style is set.
        if (getDefaultText().equals(sbSearch.getText())) {
            sbSearch.addStyleDependentName("default");
        }
    }
    */
}
